------------------------------------------------------------------------------
-- INTEL DEVELOPER'S SOFTWARE LICENSE AGREEMENT
--
-- BY USING THIS SOFTWARE, YOU ARE AGREEING TO BE BOUND
-- BY THE TERMS OF THIS AGREEMENT.  DO NOT USE THE SOFTWARE
-- UNTIL YOU HAVE CAREFULLY READ AND AGREED TO THE FOLLOWING
-- TERMS AND CONDITIONS.  IF YOU DO NOT AGREE TO THE TERMS
-- OF THIS AGREEMENT, PROMPTLY RETURN THE SOFTWARE PACKAGE
-- AND ANY ACCOMPANYING ITEMS.
--
-- IF YOU USE THIS SOFTWARE, YOU WILL BE BOUND BY THE TERMS
-- OF THIS AGREEMENT.
--
-- LICENSE:  Intel Corporation ("Intel") grants you
-- the non-exclusive right to use the enclosed software
-- program ("Software").  You will not use, copy, modify,
-- display, rent, sell or transfer the Software or any portion
-- thereof, except as provided in this Agreement.
--
-- System OEM Developers may:
-- 1.      copy the Software for internal support, backup
--         or archival purposes;
-- 2.      internally install, use, display, or distribute
--         Intel owned Software in object code format;
-- 3.      internally modify Software source code that
--         Intel makes available to you for internal use
--         only as an OEM Developer;
-- 4.      internally install, use, display, modify, distribute,
--         and/or make derivatives ("Derivatives") of Intel owned
--         Software ONLY if you are a System OEM Developer and
--         NOT an end-user.
--
-- RESTRICTIONS:
--
-- YOU WILL NOT:
-- 1.      copy the Software, in whole or in part, except as
--         provided for in this Agreement;
-- 2.      decompile or reverse engineer any Software provided
--         in object code format;
-- 3.      distribute any Software or Derivative code to any
--         end-users, unless approved by Intel in a prior writing.
--
-- TRANSFER: You may not transfer the Software to any third
-- party without Intel's prior written consent.
--
-- OWNERSHIP AND COPYRIGHT OF SOFTWARE: Title to the Software
-- and all copies thereof remain with Intel or its vendors.
-- The Software is copyrighted and is protected by United States
-- and international copyright laws.  You will not remove the
-- copyright notice from the Software.  You agree to prevent
-- any unauthorized copying of the Software.
--
-- DERIVATIVE WORK: OEM Developers that make Derivatives will
-- not be required to provide Intel with a copy of the source
-- or object code.  Any modification of Software shall be at
-- your sole risk and expense. No Software or Derivative
-- distribution to any third party is permitted under this
-- Agreement.
--
-- DUAL MEDIA SOFTWARE: If the Software package contains
-- multiple media, you may only use the medium appropriate
-- for your system.
--
-- WARRANTY: Intel warrants that it has the right to license
-- you to use, modify, display, or distribute the Software as
-- provided in this Agreement. The Software is provided "AS IS"
-- without WARRANTY of any kind.  Intel makes no representations
-- to upgrade, maintain, or support the Software at any time.
--
--
-- THE ABOVE WARRANTIES ARE THE ONLY WARRANTIES OF ANY
-- KIND, EITHER EXPRESS OR IMPLIED, INCLUDING WARRANTIES
-- OF MERCHANTABILITY OR FITNESS FOR ANY PARTICULAR PURPOSE
-- OR INFRINGEMENT OF ANY PATENT, COPYRIGHT OR OTHER
-- INTELLECTUAL PROPERTY RIGHT.
--
-- LIMITATION OF LIABILITY: NEITHER INTEL NOR ITS
-- VENDORS OR AGENTS SHALL BE LIABLE FOR ANY LOSS
-- OF PROFITS, LOSS OF USE, LOSS OF DATA, INTERRUPTION
-- OF BUSINESS, NOR FOR INDIRECT, SPECIAL, INCIDENTAL
-- OR CONSEQUENTIAL DAMAGES OF ANY KIND WHETHER UNDER
-- THIS AGREEMENT OR OTHERWISE, EVEN IF ADVISED OF THE
-- POSSIBILITY OF SUCH DAMAGES.
--
-- TERMINATION OF THIS LICENSE: Intel reserves the right
-- to conduct or have conducted audits to verify your
-- compliance with this Agreement.  Intel may terminate
-- this Agreement at any time if you are in breach of
-- any of its terms and conditions.  Upon termination,
-- you will immediately destroy, and certify in writing
-- the destruction of, the Software or return all copies
-- of the Software and documentation to Intel.
--
-- U.S. GOVERNMENT RESTRICTED RIGHTS: The Software and
-- documentation were developed at private expense and
-- are provided with "RESTRICTED RIGHTS".  Use, duplication
-- or disclosure by the Government is subject to restrictions
-- as set forth in FAR52.227-14 and DFAR252.227-7013 et seq.
-- or its successor.
--
-- EXPORT LAWS: You agree that the distribution and
-- export/re-export of the Software is in compliance
-- with the laws, regulations, orders or other restrictions
-- of the U.S. Export Administration Regulations.
--
-- APPLICABLE LAW: This Agreement is governed by the
-- laws of the State of California and the United States,
-- including patent and copyright laws.  Any claim
-- arising out of this Agreement will be brought in
-- Santa Clara County, California.
--
-- Copyright 1996, Intel Corporation, All Rights Reserved
------------------------------------------------------------------------------
-- Copyright 1997, Intel Corporation
-- Functionality and specifications based on StrataFlash(tm) - MLC EAS Rev 2.0
-- specifications (Ref. FM-0911)

-- Refer to Intel Folsom Document Control for spec.

-- Release History --
-- Date           Rev   Comments
-- July 18, 1997  0.0   Began work from Word-Wide FlashFile model


library ieee;
use std.textio.all;
use ieee.std_logic_1164.all;

-----------------------------------

entity Strat64 is

	port (
		dq	: inout std_logic_vector(15 downto 0);
		addr	: in std_logic_vector(22 downto 0);
		byteb	: in bit;
		ceb0	: in bit;
		ceb1	: in bit;
		ceb2	: in bit;	
		rpb	: in real;
		oeb	: in bit;
		web	: in bit;
		sts	: out std_logic;
		vpen	: in real;
		vcc	: in real;
		vccq	: in real
		);
end Strat64;


architecture behavior of Strat64 is

-- These define the size of the device.
-- MaxAddrLine is the number of the highest-order address line, when the byte-order line
-- is numbered A0.  A20, then, is the MaxAddrLine for a 16-meg device.

	CONSTANT MaxAddrLine	: INTEGER := 17;	-- 2 megabit


--	The real MaxAddrLine for the 64-meg part is below.  It is commented out to allow us to simulate a
--	smaller part, so our simulation runs faster.
--	CONSTANT MaxAddrLine	: INTEGER := 22;   -- Note - this is obviously size-related



	CONSTANT length		: INTEGER := (2**(MaxAddrLine) - 1);

-- These commands form the Intel Standard Command Set / Basic Command Set.
	CONSTANT ClearSRCmd            : INTEGER := 16#50# ;
	CONSTANT EraseSingleBlockCmd   : INTEGER := 16#20# ;
	CONSTANT ProgramCmd            : INTEGER := 16#10# ;
	CONSTANT Program2Cmd           : INTEGER := 16#40# ;
	CONSTANT ReadArrayCmd          : INTEGER := 16#FF# ;
	CONSTANT ReadCSRCmd            : INTEGER := 16#70# ;
	CONSTANT ReadIDCmd             : INTEGER := 16#90# ;
	CONSTANT ResumeCmd             : INTEGER := 16#D0# ;
	CONSTANT ConfirmCmd            : INTEGER := 16#D0# ;
	CONSTANT SuspendCmd            : INTEGER := 16#B0# ;

-- These commands form the Intel Extended Command Set / Scaleable Command Set.
	CONSTANT ReadQueryCmd		: INTEGER := 16#98# ;
	CONSTANT WriteBufferCmd		: INTEGER := 16#E8# ;
	CONSTANT STSConfigCmd		: INTEGER := 16#B8# ;
	CONSTANT SetBlkLockbitCmd	: INTEGER := 16#60# ;
	CONSTANT SetBlkLockbitCmd2	: INTEGER := 16#01# ;
	CONSTANT SetMstrLockbitCmd2	: INTEGER := 16#F1# ;
	CONSTANT ClrBlkLockbitCmd	: INTEGER := 16#60# ;
	CONSTANT FullChipEraseCmd	: INTEGER := 16#30# ;


-- These are values for the ReadType variable.
	CONSTANT ReadArray	: INTEGER := 1;
	CONSTANT ReadIDCodes	: INTEGER := 2;
	CONSTANT ReadStatReg	: INTEGER := 3;
	CONSTANT ReadQuery	: INTEGER := 4;
	CONSTANT ReadXStatReg	: INTEGER := 5;
	CONSTANT NoReadPending	: INTEGER := 0;

-- These are values for the WriteType variable.
-- They are also used for the Running variable, which stores the type of operation currently
-- occupying the device.
	CONSTANT WriteCmd		: INTEGER := 0;
	CONSTANT WriteProgram		: INTEGER := 1;
	CONSTANT WriteBlkErase		: INTEGER := 2;
	CONSTANT WriteBufNum		: INTEGER := 3;
	CONSTANT WriteSTSPinCfg		: INTEGER := 4;
	CONSTANT WriteLockBit		: INTEGER := 5;
	CONSTANT WriteFullChipErase	: INTEGER := 6;
	CONSTANT WriteClrLockBit	: INTEGER := 7;
	CONSTANT SuspensionLatency 	: INTEGER := 8;
-- Note:  While SuspensionLatency isn't an operation or a valid value for WriteType, it is
-- placed here because it is a valid value for the Running variable.  When an operation is
-- suspended, the device acts as it does when a program or erase is executed; the status 
-- register is placed on the bus during read cycles, and SR.7 and STS are asserted (assuming
-- STS is in RY/BY# mode) until the suspension latency time is elapsed. 
	CONSTANT ResetLatency		: INTEGER := 9;
	CONSTANT WriteBufData		: INTEGER := 10;
	CONSTANT WriteBufCfrm		: INTEGER := 11;


-- These are the voltage ranges in the 5 volt spec (28FxxxS5).
	CONSTANT Vcc5v10pcMin		: REAL	:= 4.50;
	CONSTANT Vcc5v10pcMax		: REAL	:= 5.50;
	CONSTANT Vccq5vMin		: REAL	:= 4.5;
	CONSTANT Vccq5vMax		: REAL	:= 5.5;
	CONSTANT Vccq3vMin		: REAL	:= 2.7;
	CONSTANT Vccq3vMax		: REAL	:= 3.6;
	CONSTANT Vpp5vLockout		: REAL	:= 3.6;
	CONSTANT Vpp5vMin		: REAL	:= 4.5;
	CONSTANT Vpp5vMax		: REAL	:= 5.5;
	CONSTANT VhhMin			: REAL	:= 11.4;
	CONSTANT VhhMax			: REAL	:= 12.6;
	CONSTANT VihMin			: REAL	:= 2.0;


-- These specs are the 5V AC write specs.  
	constant TPHWL	: time := 1000 ns;
	constant TELWL	: time := 25 ns;
	constant TWLEL	: time := 0 ns;
	constant TWLWH	: time := 70 ns;
	constant TELEH	: time := 70 ns;
	constant TSHWH	: time := 100 ns;
	constant TVPWH	: time := 0 ns;
	constant TAVWH	: time := 50 ns;
	constant TDVWH	: time := 50 ns;
	constant TWHDX	: time := 0 ns;
	constant TWHAX	: time := 0 ns;
	constant TEHAX	: time := TWHAX;
	constant TEHDX	: time := TWHDX;
	constant TWHEH	: time := 10 ns;
	constant TEHWH	: time := 10 ns;
	constant TWHWL	: time := 30 ns;
	constant TEHEL	: time := 30 ns;
	constant TWHRL	: time := 90 ns;
	constant TWHGL	: time := 35 ns;
	constant TQVVL	: time := 0 ns;
	constant TQVSL	: time := 0 ns;  

	constant TPHEL	: time := TPHWL;
	constant TSHEH	: time := TSHWH;
	constant TDVEH	: time := TDVWH;
	constant TAVEH	: time := TAVWH;
	constant TVPEH	: time := TVPWH;

	constant TPLPH	: time := 100 ns;
	constant TPLRH	: time := 12 us;


-- These times are the latency times.  They specify how long each operation leaves the
-- device busy (as manifested by SR.7 and STS in the appropriate mode).  Adding NOW to
-- these times gives a time of completion in the future - comparing subsequent NOWs to 
-- that time will allow us to note when an operation finishes (and then update the Running,
-- SR.7, and STS variables).
	constant W16_ProgByteWBuffer :	time := 6 us;
	constant W16_ProgWordWBuffer :	time := 12 us;
   
-- This is the real time to program a byte directly.  It's too long to use practically in a VHDL model.
--	constant W16_ProgByteWOBuffer:	time := 120 us;   -- tWHQV1 / tEHQV1
-- This is a fake time which works better.
	constant W16_ProgByteWOBuffer:	time := 10 us;

-- This is the real time to program a word directly.  It's too long to use practically in a VHDL model.
--	constant W16_ProgWordWOBuffer:	time := 120 us;   -- tWHQV1 / tEHQV1
-- This is a fake time which works better.
	constant W16_ProgWordWOBuffer:	time := 10 us;


-- This is the real Block Erase time.  It's too long to use practically in a VHDL model.
--	constant W16_BlockErase:   time := 1.5 sec;   -- tWHQV2 / tEHQV2

-- This is a fake BlockErase time to fit simulation capabilities
	constant W16_BlockErase:	time := 10 us;

-- This is the real Full Chip Erase time.  It's too long to use practically in a VHDL model.
-- It's also a complete guess, as the real time doesn't show in the spec.
--	constant W16_FullChipErase:	time := 10.7 sec;   -- tWHQVx / tEHQVx

-- This is a fake Full Chip Erase time to fit simulation capabilities.
	constant W16_FullChipErase:	time := 100 us;

	constant W16_SetLockBit:   time := 12 us;   -- tWHQV3 / tEHQV3

-- This is the real Clear Lockbits time.  You get the idea.
--	constant W16_ClearLockBit: time := 1.5 sec;   -- tWHQV4 / tEHQV4

-- This is a fake ClearLockBit time to fit simulation capabilities
	constant W16_ClearLockBit: time := 10 us;

	-- For-real.
	constant W16_EraseSuspToRead: time := 25 us;   -- tWHRH2 / tEHRH2

	
-- These are the values for state and new-state.  They help us keep track of what's happening
-- on the bus, and what sigs and transitions we're looking at.
	constant Powerup	: integer := 0;
	constant OutputDisable	: integer := 1;
	constant Powerdown	: integer := 2;
	constant ReadCycleZ	: integer := 3;
	constant ReadCycleX	: integer := 4;
	constant ReadCycleV	: integer := 5;
	constant WriteCycle	: integer := 8;
	constant WEActive	: integer := 9;

-- These constants work with the SRAM write buffer subsystem.
	constant BufFree	: integer := 0;
	constant BufReading	: integer := 1;
	constant BufFull	: integer := 2;
	constant BufWriting	: integer := 3;
	constant BufSuspended	: integer := 4;

	constant NumBuffers	: integer := 1;
	constant ByteMaxBufWrite	: integer := 31;
	constant WordMaxBufWrite	: integer := 15;


-- This signal exists because it is convenient to view Vpp as a logic signal rather than a
-- continuous variable to calculate Vpp setup and hold times.

	SIGNAL VppSig : STD_LOGIC;
	SIGNAL VccSig : STD_LOGIC;

	SIGNAL rpbsig : bit;
	SIGNAL rpbhighsig : bit;

-- This signal exists to clock the main state machine.
	SIGNAL clock1 : bit := '0';
	SIGNAL clock2 : bit := '1';

	SIGNAL ceb : BIT;

--***************************************************************

TYPE word IS ARRAY(15 downto 0) OF STD_ULOGIC;
TYPE memarray IS ARRAY(0 TO length) OF word;
TYPE BlockArray IS ARRAY (2**(MaxAddrLine-15) downto 0) OF word;
TYPE QueryTableArray IS ARRAY (0 to 16#3E#) OF word;
TYPE BufCountType IS ARRAY (1 TO NumBuffers) OF integer;
TYPE BufAddrArrayType IS ARRAY (1 TO NumBuffers) OF integer;
TYPE WriteBuffersType IS ARRAY (1 TO NumBuffers, 0 TO WordMaxBufWrite) OF word;
TYPE BufStatusType IS ARRAY (1 TO NumBuffers) OF integer;
TYPE BufbytebLatchType IS ARRAY (1 TO NumBuffers) OF bit;

BEGIN

	MAIN:PROCESS(clock2)

	-- Scratchpad for converting STD_LOGIC_VECTOR to INTEGER
	VARIABLE byteadder :INTEGER := 0;
	VARIABLE adder :INTEGER := 0;
	VARIABLE twosum   :INTEGER := 1;

	-- Scratchpad to handle the various permutations of orphaned bytes
	VARIABLE j_start :INTEGER := 0;
	VARIABLE j_end :INTEGER := 0;

	-- Scratchpad to count down the remaining write cycles in a buffered write
	VARIABLE countdown :INTEGER; 

	-- Integer copies of the latched data and the low byte of the latched data
	VARIABLE data : INTEGER;
	VARIABLE datalo :INTEGER;

	-- The entire memory
	VARIABLE memory :memarray;

	-- An array of the blocks' status bytes.  Read by the Read Query xxxx02 addresses
	VARIABLE BlockStatus : BlockArray;
	VARIABLE MstrLockBit : std_ulogic := '0';	


	-- Scratchpad for the integer value of the block number of the latched address
	VARIABLE BlockNum : INTEGER;           

	-- Scratchpad to hold a pseudo-latched value of the BYTE# line.  
	-- Used during reads to watch for a transition of that line, to verify timing
	-- parameters tFLQV, tFHQV, and tFLQZ.  tFLQZ isn't implemented correctly anyway.
	VARIABLE latchbyteb :BIT;

	-- The Status Register goes here.
	VARIABLE SR :STD_LOGIC_VECTOR (7 downto 0) := "10000000";

	-- The Extended Status Register goes here.
	VARIABLE XSR :STD_LOGIC_VECTOR (7 downto 0) := "1XXXXXXX";
	
	-- This is used to keep track of what information the device will give on the next
	-- READ bus cycle.  See the table in the constants section for examples.
	VARIABLE ReadType :INTEGER := ReadArray;

	-- This is used to keep track of what information the device expects to receive on 
	-- the next WRITE bus cycle.  See the table in the constants section for examples.
	VARIABLE WriteType :INTEGER := WriteCmd;

	-- This is used to store the state the main state machine of the model will enter at
	-- the next pseudo-clock (0.5 ns).
	VARIABLE next_state :INTEGER := 0;

	-- This stores the current state of the main state machine of the model.
	VARIABLE state:INTEGER := Powerup;

	-- Scratchpad to compare the value on the address bus to the latched address, to 
	-- watch for transitions.  Perhaps unnecessary.
	VARIABLE equal :BIT;

	-- One-timer used to initialize all the other variables.
	VARIABLE loader :BIT:= '0';

	-- Latched data and address values go here
	VARIABLE hold_data : word;
	VARIABLE hold_add : STD_ULOGIC_VECTOR (MaxAddrLine downto 0);

	-- When an operation is suspended, the latched data and address values before the
	-- suspension go here.
	VARIABLE susp_add : STD_ULOGIC_VECTOR (MaxAddrLine downto 0);

	-- These variables store the time of the last transition of the respective line
	-- to the respective state.  This is necessary to verify pulse width timing
	-- requirements, those which measure the time between successive transitions of the
	-- same signal.
	VARIABLE timeWH : TIME := 0 ns;
	VARIABLE timeWL : TIME := 0 ns;
	VARIABLE timeEH : TIME := 0 ns;
	VARIABLE timeEL : TIME := 0 ns;
	VARIABLE timeSRValid : TIME := 0 ns;
	VARIABLE timeRPL : TIME := 0 ns;

	-- This variable holds the information in the Common Flash Interface database.
	VARIABLE QueryTable : QueryTableArray; 

	-- This holds the constant denoting what, if any, operation is currently suspended.
	-- See the constants section for why this needs to be rewritten.
	VARIABLE SuspendedOp : INTEGER := 0;

	-- This holds the WriteType constant of the operation currently busying the device.
	VARIABLE Running : INTEGER := 0;

	-- This holds the future time of the pending completion of the operation noted in
	-- Running.
	VARIABLE Completion : TIME := 0 ns;

	-- This holds the last byte written with the STS Mode Configuration command.
	-- It serves to store the current mode of the STS output pin.
	VARIABLE STSMode : STD_LOGIC_VECTOR (1 downto 0);

	-- This variable holds the value of the SR when we move from ReadStateX to ReadStateV.
	-- This is important for compliance to spec, because it explicitly mentions that the SR is latched
	-- upon read, and is not refreshed until OE or CE transitions.
	VARIABLE SRLatch : STD_LOGIC_VECTOR (7 downto 0);

	-- These variables hold the block number of the block being erased, and the address of the word
	-- being programmed.  If we attempt to read from either of these locations, we return Xes.
	VARIABLE ErasingBlk : INTEGER;
	VARIABLE ProgrammingAddr : INTEGER;

	-- These variables have to do with the SRAM write buffering system.
	-- Forcing all arrays to be in type declarations may be "good" style, but it's silly
	-- for things that show up once.  Thank you.  That is all.

	VARIABLE BufCount : BufCountType;
	VARIABLE BufBytebLatch : BufBytebLatchType;
	VARIABLE BufStartAddr : BufAddrArrayType;
	VARIABLE BufEndAddr : BufAddrArrayType;
	VARIABLE BufBlockNum : BufAddrArrayType;
	VARIABLE WriteBuffers : WriteBuffersType;
	VARIABLE BufStatus : BufStatusType;
	VARIABLE ActiveBuffer : integer;

	-- These are flags to determine if a hold time error has been reported.  Bad form to report
	-- the same error every billionth of a second.
	VARIABLE AddrHoldFlag 	: bit := '0';
	VARIABLE DataHoldFlag 	: bit := '0';
	VARIABLE VppHoldFlag 	: bit := '0';
	VARIABLE WPHoldFlag 	: bit := '0';
	VARIABLE VppMaintLFlag 	: bit := '0';
	VARIABLE VppMaintSFlag 	: bit := '0';
	VARIABLE VccMaintLFlag	: bit := '0';
	VARIABLE VccMaintSFlag	: bit := '0';
	
	-- These are the read timing specs.  They became variables to allow multiple Vcc ranges in a given
	-- model.

	VARIABLE TAVQV              : time;
	VARIABLE TELQV              : time;
	VARIABLE TPHQV              : time;
	VARIABLE TGLQV              : time;
	VARIABLE TELQX              : time;
	VARIABLE TEHQZ              : time;
	VARIABLE TGLQX              : time;
	VARIABLE TGHQZ              : time;
	VARIABLE TOH                : time;
	VARIABLE TELFL              : time;
	VARIABLE TELFH              : time;
	VARIABLE TFLQV              : time;
	VARIABLE TFHQV              : time;
	VARIABLE TFLQZ              : time;

	-- This is the max value for Vih
	VARIABLE VihMax		: real;



-- MAIN ROUTINE STARTS HERE!

	BEGIN

	ceb <= (ceb0 AND ceb1) OR (NOT(ceb2) AND (ceb1 XOR ceb0));		-- ceb is strange

	-- VppSig is high if Vpp is in the valid range

	IF ((Vpen >= Vpp5vMin) AND (Vpen <= Vpp5vMax)) THEN 
		VppSig <= '1';
	ELSE  VppSig <= '0';
	END IF;


	-- VccSig is high if Vcc is in the valid range
	IF ((vcc >= Vcc5v10pcMin) AND (vcc <= Vcc5v10pcMax)) THEN
		VccSig <= '1';
	ELSE	VccSig <= '0';
	END IF;

	IF ((Vccq >= Vccq5vMin) AND (Vccq <= Vccq5vMax)) THEN				-- 32 M 120/170 part 
		tAVQV := 150 ns;
		tELQV := 150 ns;
		tPHQV := 210 ns;
		tGLQV := 50 ns;
		tELQX := 0 ns;
		TEHQZ := 55 ns;
		tGLQX := 0 ns;
		TGHQZ := 15 ns;
		TOH   := 0 ns;
		TELFL := 10 ns;
		TELFH := 10 ns;
		TFLQV := 1000 ns;
		TFHQV := 1000 ns;
		TFLQZ := 1000 ns;

	ELSIF ((Vccq >= Vccq3vMin) AND (Vccq <= Vccq3vMax)) THEN 
		tAVQV := 200 ns;
		tELQV := 200 ns;
		tPHQV := 260 ns;
		tGLQV := 100 ns;
		tELQX := 0 ns;
		TEHQZ := 105 ns;
		tGLQX := 0 ns;
		TGHQZ := 65 ns;
		TOH   := 0 ns;
		TELFL := 10 ns;
		TELFH := 10 ns;
		TFLQV := 1000 ns;
		TFHQV := 1000 ns;
		TFLQZ := 1000 ns;


	ELSE VccSig <= '0';
	END IF;

	VihMax := Vcc;

	IF ((rpb >= VihMin) AND (rpb <=VihMax)) THEN
		rpbhighsig <= '0';
		rpbsig <= '1';
	ELSIF ((rpb >= VhhMin) AND (rpb <= VhhMax)) THEN
		rpbhighsig <= '1';
		rpbsig <= '1';
	ELSE 
		rpbhighsig <= '0';
		rpbsig <= '0';
	END IF; 

	

-- This section intializes the memory array which contains all 1's.
-- It also initializes the BlockStatus memory, marking each block unlocked,
-- and sets up the CFI Query table.

	IF (loader = '0') THEN

		FOR i IN 0 TO 2**(MaxAddrLine - 16) LOOP
			BlockStatus(i) := "0000000000000000";
		END LOOP;

		FOR i IN 0 TO length LOOP
			memory(i) := "1111111111111111";
		END LOOP;

		FOR i IN 1 TO NumBuffers LOOP
			BufStatus(i) := BufFree;
		END LOOP;


-- REVIEW THIS TABLE BEFORE RELEASE.
-- This section initializes the CFI Query database table.
		QueryTable(16#0#)  := "0000000010001001"; -- MfrId
		QueryTable(16#1#)  := "0000000000010101"; -- Device Code
		
		QueryTable(16#10#) := "0000000001010001"; -- "Q"   
		QueryTable(16#11#) := "0000000001010010"; -- "R"
		QueryTable(16#12#) := "0000000001011001"; -- "Y"
		
		QueryTable(16#13#) := "0000000000000001"; 
		QueryTable(16#14#) := "0000000000000000";

		QueryTable(16#15#) := "0000000000110001"; -- $31 = location of
		QueryTable(16#16#) := "0000000000000000"; --       extended qry table

		QueryTable(16#17#) := "0000000000000000";
		QueryTable(16#18#) := "0000000000000000";

		QueryTable(16#19#) := "0000000000000000";
		QueryTable(16#1A#) := "0000000000000000";

		QueryTable(16#1B#) := "0000000001000101"; -- Vcc Min Write/Erase Voltage
		QueryTable(16#1C#) := "0000000001010101"; -- Vcc Max Write/Erase Voltage
		QueryTable(16#1D#) := "0000000000000000"; -- Vpp Min Write/Erase Voltage
		QueryTable(16#1E#) := "0000000000000000"; -- Vpp Max Write/Erase Voltage

		QueryTable(16#1F#) := "0000000000000111";    
		QueryTable(16#20#) := "0000000000000111";
		QueryTable(16#21#) := "0000000000001010";
		QueryTable(16#22#) := "0000000000000000";

		QueryTable(16#23#) := "0000000000000100";    
		QueryTable(16#24#) := "0000000000000100";    
		QueryTable(16#25#) := "0000000000000100";    
		QueryTable(16#26#) := "0000000000000000";    

		QueryTable(16#27#) := "0000000000010111"; -- Device size = 2^n (64-meg)
		QueryTable(16#28#) := "0000000000000010"; -- x8/x16 async
		QueryTable(16#29#) := "0000000000000000";
		
		QueryTable(16#2A#) := "0000000000000101"; -- Max # bytes in multiprogram
		QueryTable(16#2B#) := "0000000000000000"; -- = 2^n

		QueryTable(16#2C#) := "0000000000000001"; -- # of regions of erase blks

		QueryTable(16#2D#) := "0000000000101111"; -- # of erase blocks in reg 1 LB
		QueryTable(16#2E#) := "0000000000000000"; -- # of erase blocks in reg 1 HB
		QueryTable(16#2F#) := "0000000000000000"; -- 256n = size of each blk LB
		QueryTable(16#30#) := "0000000000000010"; -- 256n = size of each blk HB
		
-- Intel-specific Extended Query Table starts here

		QueryTable(16#31#) := "0000000001010000"; -- "P"
		QueryTable(16#32#) := "0000000001010010"; -- "R"
		QueryTable(16#33#) := "0000000001001001"; -- "I"

		QueryTable(16#34#) := "0000000000110001"; -- Major ver no.
		QueryTable(16#35#) := "0000000000110001"; -- Minor ver no.
		
		QueryTable(16#36#) := "0000000000001010"; -- Optional feature support
		QueryTable(16#37#) := "0000000000000000"; -- Optional feature support
		QueryTable(16#38#) := "0000000000000000"; -- Optional feature support
		QueryTable(16#39#) := "0000000000000000"; -- Optional feature support

		QueryTable(16#3A#) := "0000000000000001"; -- Supported fns after susp.
		
		QueryTable(16#3B#) := "0000000000000001"; -- Block Status Register Mask
		QueryTable(16#3C#) := "0000000000000000"; -- more BSR Mask

		QueryTable(16#3D#) := "0000000001010000"; -- Vcc Optimum Wr/Erase Voltage
		QueryTable(16#3E#) := "0000000000000000"; -- Vpp Optimum Wr/Erase Voltage

		SR := "10000000";
		sts <= 'Z';

		loader := '1';
	END IF;
-- Setup one-shot stuff ends here

		  

-- One implicit assumption of the state machine is that no two control signals will
-- transition within half a nanosecond of each other.  It simplifies design greatly to
-- assume that only one transition occurs during any given pseudo-clock.


	CASE state IS


	WHEN Powerup =>		-- This state corresponds to CE, WE, OE, and RP all high (inactive) 

		dq <= "ZZZZZZZZZZZZZZZZ";					-- Outputs are off
		
		IF (rpbsig = '0') THEN						-- RP is asserted
			TimeRPL := NOW;						-- Note when rpb went low
			next_state := Powerdown;				-- Go to Powerdown
    
		ELSIF (ceb = '0') THEN						-- CE is asserted
			timeEL := NOW;						-- Note when ceb went low
			next_state := OutputDisable;				-- Go to OutputDisable

		ELSIF (web = '0') THEN						-- WE is asserted
			timeWL := NOW;						-- Note when web went low
			next_state := WEActive;					-- Go to WEActive
		ELSE
			next_state := Powerup;					-- Nothing happened
		END IF;
		
	-- END STATE Powerup
	


	WHEN OutputDisable =>	-- This state has CE low/asserted, and WE, OE, and RP high/unasserted.
				-- From here, we can do a ReadCycle, a WriteCycle, a PowerDown, or
				-- even go back to PowerUp.
		
		IF (rpbsig = '0') THEN						-- RP is asserted
			TimeRPL := NOW;						-- note when RP went low
			next_state := Powerdown;				-- Go to PowerDown 


		ELSIF (ceb = '1') THEN						-- CE is unasserted
			timeEH := NOW;						-- Note when CE went high

			ASSERT (web'LAST_EVENT >= tWHEH)			-- Check how long from WE
			REPORT "Enable Hold violation (tWHEH)."			-- going high to CE going high
			SEVERITY ERROR;						-- Enable Hold time

			next_state := Powerup;					-- Return to Powerup


		ELSIF (oeb = '0') THEN						-- OE is asserted - we're
										-- beginning a read cycle.
			next_state := ReadCycleZ;				-- Go to ReadCycleZ state 
										-- and wait for outputs to 
										-- turn on

		ELSIF (web = '0') THEN						-- WE is asserted - we're
										-- beginning a write cycle.

			timeWL := NOW;						-- Note when WE went low

			ASSERT (ceb'LAST_EVENT >= tELWL)			-- Verify that CE has been 
			REPORT "Enable Setup violation (tELWL)"			-- asserted long enough -
			SEVERITY ERROR;						-- Enable Setup time

			ASSERT (rpbsig'LAST_EVENT >= tPHWL)			-- Verify that RP has been 
			REPORT "Powerdown Recovery Time violation (tPHWL)"	-- unasserted long enough -	
			SEVERITY ERROR;						-- Powerdown Recovery time

			ASSERT (((NOW - web'LAST_EVENT) - timeWH) >= tWHWL)	-- Verify that the WE line
			REPORT "WE Pulse High Time violation (tWHWL)"		-- was high long enough -
			SEVERITY ERROR;						-- WE Pulse High time

			next_state := WriteCycle;				--Go to WriteCycle

		ELSE
			next_state := OutputDisable;				--Nothing happened

		END IF;

	-- END STATE OutputDisable




	-- This state corresponds to WE# asserted (low), and OE# and CE# unasserted (high).
	-- The only reason to come here is to set up or leave a CE-controlled write.

	WHEN WEActive =>	-- WE# Asserted State - for CE-controlled writes
		
		IF (rpbsig = '0') THEN						-- RP is asserted
			TimeRPL := NOW;						-- note when rpb became asserted
			next_state := Powerdown; 				-- Goes to Powerdown

		ELSIF (web = '1') THEN						-- WE is unasserted again
										-- Go back to Powerup 
			timeWH := NOW;						-- Note when web went high

			ASSERT (ceb'LAST_EVENT >= tEHWH)			-- Verify there was enough time 
			REPORT "Enable Hold violation (tEHWH)."			-- from CE high to WE high -
			SEVERITY ERROR;						-- Enable Hold time

			next_state := Powerup;      				-- Returns to powerup


		ELSIF (ceb = '0') THEN						-- CE is asserted - about to
										-- do a WriteCycle

			timeEL := NOW;						-- Note when CE went low

			ASSERT (web'LAST_EVENT >= tWLEL)			-- Verify WE was low long enough
			REPORT "Enable Setup violation (tWLEL)"			-- before CE went low -
			SEVERITY ERROR;						-- Enable Setup time

			ASSERT (rpbsig'LAST_EVENT >= tPHEL)			-- Verify RP was unasserted long
			REPORT "Powerdown Recovery Time violation (tPHEL)"	-- enough before CE went low -
			SEVERITY ERROR;						-- Powerdown Recovery time

			ASSERT (((NOW - ceb'LAST_EVENT) - timeEH) >= tEHEL)	-- Verify CE was high long 
			REPORT "CE Pulse High Time violation (tEHEL)"		-- enough - CE Pulse Width High
			SEVERITY ERROR;	

			next_state := WriteCycle;				-- Goes to WriteCycle

		ELSE
			next_state := WEActive;					-- Nothing happened

		END IF;

	-- END STATE WEActive

	

	WHEN ReadCycleZ =>	-- OE and CE either just became asserted or just became unasserted.
				-- We're in that portion of a read cycle right before the outputs
				-- turn on or right after they shut off

		DQ <= "ZZZZZZZZZZZZZZZZ";				-- All high-Z on the data outputs
		ASSERT (web'LAST_EVENT >= tWHGL)			-- Verify that WE hasn't been asserted in
		REPORT "Violation of tWHGL (Write recovery before read)."
		SEVERITY ERROR;						-- awhile - Write Recovery Before Read
	

		IF (rpbsig = '0') THEN					-- rpb became asserted - go to Powerdown
			TimeRPL := NOW;					-- note when rpb became asserted
			next_state := PowerDown;	

		ELSIF ((ceb = '0' AND ceb'LAST_EVENT >= tELQX) AND	-- If all the hold time requirements
			 (oeb = '0' AND oeb'LAST_EVENT >= tGLQX)) THEN	-- are met, turn on the outputs with
			  next_state := ReadCycleX;			-- invalid data - go to ReadCycleX
			  SRLatch := SR;				-- and latch the SR.
		


		ELSIF (ceb = '1') THEN next_state := PowerUp;		-- CE became unasserted somehow - 
									-- go to Powerup

		ELSIF (oeb = '1') THEN next_state := OutputDisable;	-- OE became unasserted - we're on
									-- the back edge of a read cycle.
									-- Go to OutputDisable

		ELSE next_state := ReadCycleZ;				-- Nothing important happened -
									-- stay where we are.
		END IF; 

	-- END STATE ReadCycleZ



	WHEN ReadCycleX =>	-- This state corresponds to the two places on the read cycle timing diagram
				-- where the data is hatched to show that it isn't valid yet.  Either we're
				-- just beginning a read, or we're just leaving it.  From here, we go either
				-- to ReadCycleV, if the data's about to become valid, or to ReadCycleZ, if
				-- the outputs are about to shut off.


		IF (byteb = '1') THEN DQ <= "XXXXXXXXXXXXXXXX";	-- If the device is in word mode, place all
								-- Xes on the outputs.

		ELSE 		      DQ <= "ZZZZZZZZXXXXXXXX";	-- If the device is in byte mode, place all
								-- Xes on the low byte of the output and
								-- leave the high byte off.

		END IF;


		IF (rpbsig = '0') THEN				-- rpb asserted - go to Powerdown state 
			TimeRPL := NOW;
			next_state := PowerDown;

		ELSIF (addr'LAST_EVENT >= tAVQV AND rpbsig'LAST_EVENT >= tPHQV AND
			 ceb'LAST_EVENT >= tELQV AND byteb'LAST_EVENT >= tFLQV AND
			 oeb'LAST_EVENT >= tGLQV) THEN		-- If all the setup conditions are met, 
								-- place valid data on the outputs and go to
								-- ReadCycleV

			FOR j IN MaxAddrLine downto 0 LOOP	-- Latch the address that's on the bus when
				hold_add(j) := addr(j);		-- we go to ReadCycleV, to check to see if
			END LOOP;				-- it changes; if it does, the data goes 
								-- invalid.
			next_state := ReadCycleV;


		ELSIF ((oeb = '1' AND oeb'LAST_EVENT >= tGHQZ)	-- If one of the control lines has gone high,
			OR (ceb = '1' AND			-- go the other way - turn off the data bus,
			ceb'LAST_EVENT >= tEHQZ)) THEN		-- and go to ReadCycleZ (after a certain lag
			next_state := ReadCycleZ;		-- time)

		ELSE next_state := ReadCycleX;			-- Nothing happened.  

		END IF;
	-- END STATE ReadCycleX



	WHEN ReadCycleV =>	-- This is the moment we've been waiting for - in ReadCycleV,
				-- all the setup conditions are finally met, and we place
				-- valid data on the data outputs.  The only place to go from here
				-- is to ReadCycleX, when one of the control inputs changes 
				-- (oeb, ceb, and addr all cause this)

		adder := 0;						-- This is generic code to take
		twosum := 1;						-- the address (held in the latch
		FOR j IN 1 to MaxAddrLine LOOP				-- we closed when transitioning
			IF (hold_add(j) = '1') THEN			-- from ReadCycleX), and calculate
				adder := adder + twosum;		-- the corresponding integer address
			END IF;						-- value in adder.
			twosum := twosum * 2;				-- Remember, hold_add is an array of
		END LOOP;						-- STD_ULOGIC.
									-- As with all addresses, it's the 
									-- WORD address unless noted - A0
									-- is ignored.
		


		CASE ReadType IS					-- This case statement splits us into
									-- the different read modes.
									-- The value of ReadType is set
									-- through sending various commands
									-- to the device.


		WHEN ReadArray =>					-- Reading from the memory itself

			BlockNum := 0;					-- Figure the number of the
			twosum := 1;					-- block to which the latched address
			FOR j IN 17 TO MaxAddrLine LOOP			-- corresponds, and store it in 
				IF hold_add(j) = '1' THEN		-- BlockNum
					BlockNum := BlockNum + twosum;
				END IF;
				twosum := twosum * 2;
			END LOOP;


			hold_data := memory(adder);			-- Place the contents of the mem
									-- in a holding area

			IF (SR(6) = '1') AND BlockNum = ErasingBlk THEN	-- If the read is in a block with
				hold_data := "XXXXXXXXXXXXXXXX";	-- erase suspended, the value's
			END IF;						-- anyone's guess

			IF (SR(2) = '1') THEN				-- If there's a write suspended,

				ActiveBuffer := 0;			-- check to see if a buffer write
									-- is what's suspended.
				
				FOR j IN 1 TO NumBuffers LOOP
					IF BufStatus(j) = BufSuspended THEN
						ActiveBuffer := j;
					END IF;
				END LOOP;

				IF ActiveBuffer /= 0  THEN		-- If it's a buffer write suspended,

					IF BufbytebLatch(ActiveBuffer) = '0' THEN	-- It was a byte-mode
											-- write to buffer
						IF (2*adder + 1 >= BufStartAddr(ActiveBuffer)) AND
						   (2*adder     <= BufEndAddr(ActiveBuffer)) THEN
						hold_data := "XXXXXXXXXXXXXXXX";
						END IF;
					ELSE	IF (adder >= BufStartAddr(ActiveBuffer) AND
						    adder <= BufEndAddr(ActiveBuffer)) THEN
						hold_data := "XXXXXXXXXXXXXXXX";
						END IF;
					END IF;
				ELSE	IF adder = ProgrammingAddr	-- It's a simple program suspended.
					THEN hold_data := "XXXXXXXXXXXXXXXX";
					END IF;
				END IF;
			END IF;



			IF (byteb = '0') THEN				-- If we're in byte mode, do more.

				IF (hold_add(0) = '0') THEN		-- Reading the low byte:  Turn off
					FOR j IN 15 DOWNTO 8 LOOP	-- the high byte in the holding area
						hold_data(j) := 'Z';
					END LOOP;			-- and that's it.
	
				ELSE					-- Reading the high byte:
					FOR j IN 15 DOWNTO 8 LOOP	-- Move the data in the high byte of
						hold_data(j-8) := hold_data(j);	-- the holding area to the
									-- low byte, 
						hold_data(j) := 'Z';	-- and turn off the high byte.
					END LOOP;
				END IF;					
			END IF;				


		-- END ReadArray subcase                		-- Holding area's set up - we're done





		WHEN ReadIDCodes =>					-- Reading from a subset of the 
									-- query database (the ID Codes)

			IF (not ((adder < 3) OR ((adder mod 65536) = 2)) OR
				((adder mod 65536) = 3)) THEN		-- This is admittedly ugly.
									-- If this test fails, we're off
									-- the memory map defined for 
									-- ReadIDCodes, and we return
				hold_data := "XXXXXXXXXXXXXXXX";	-- invalid data.

			ELSIF (adder mod 65536) = 2 THEN		-- If the address ends in 2,
									-- we're looking at the Block Status
									-- Registers.

				hold_data := BlockStatus((adder - (adder mod 65536))/65536);
									-- That's an ugly way to do an
									-- integer divide - returns the block
									-- number corresponding to the
									-- address in the latch.
									-- The holding area is loaded with
									-- the value in the Block Status Reg


				FOR j IN 15 DOWNTO 2 LOOP		-- Bits 15 through 3 of the Block
					hold_data(j) := 'X';		-- Status registers are undefined -
				END LOOP; 				-- we set them to X.

			ELSIF (adder mod 65536) = 3 THEN		-- If the address ends in 3,
									-- we're looking at the Master 
									-- Lock Bit
				FOR j IN 15 DOWNTO 1 LOOP
					hold_data(j) := 'X';
				END LOOP;
				IF MstrLockBit = '1' THEN hold_data(0) := '1';
							ELSE hold_data(0) := '0';
				END IF;
		

			ELSE	hold_data := QueryTable(adder);		-- Otherwise, we're reading the
									-- CFI Query Table, and we move the
									-- data from there to the holding area
			END IF;


			IF (byteb = '0') THEN				-- If we're in byte mode, turn off 
				FOR j IN 15 downto 8 LOOP		-- the top half of the holding area
					hold_data(j) := 'Z';		-- Nothing more intricate needed, as
				END LOOP;				-- for ID Codes, there isn't any
			END IF;						-- data in the top byte anyway


		-- END ReadIDCodes subcase


		WHEN ReadStatReg =>					-- Reading the Status Register,
									-- regardless of the address
	

			FOR j IN 15 DOWNTO 8 LOOP			-- Leave the top byte of the holding
				hold_data(j) := 'Z';			-- area turned off.
			END LOOP;

			FOR j IN 7 DOWNTO 0 LOOP			-- If SR.7 = 1, then the data in the
				IF (j = 7) OR (SRLatch(7) = '1') THEN	-- SR is valid, and we put it all in
					hold_data(j) := SRLatch(j);	-- the holding area.  If SR.7 = 0, 
				ELSE	hold_data(j) := 'Z';		-- the SR is not valid (except for 
				END IF;					-- bit 7), and we set it to Xes.
			END LOOP;

		-- END ReadStatReg subcase



		WHEN ReadXStatReg =>					-- Reading the very tiny Extended
									-- Status Register

			FOR j IN 15 DOWNTO 8 LOOP			-- Leave the top byte of the holding
				hold_data(j) := 'Z';			-- area turned off.
			END LOOP;

			hold_data(7) := XSR(7);				-- Set the 7 bit of the holding area
									-- to the value in the XSR.
			
			IF (running = 0) THEN
				FOR j IN 6 DOWNTO 0 LOOP		-- Everything else is undefined, and
					hold_data(j) := 'X';		-- thus set to Xes.
				END LOOP;
			ELSE	FOR j IN 6 DOWNTO 0 LOOP
					hold_data(j) := 'Z';
				END LOOP;
			END IF;	

		-- END ReadXStatReg subcase


		WHEN ReadQuery =>					-- Reading from the full query dbase

			IF (not ((adder < 16#3F#) OR ((adder mod 65536) = 2) OR ((adder mod 65536) = 3))) 
				THEN hold_data := "XXXXXXXXXXXXXXXX";	-- See the corresponding code in the
									-- ReadIDCodes case - if we're off
									-- the memory map, the holding area
									-- is full of undefined data.


			ELSIF (adder mod 65536) = 2 THEN		-- If we're in the Block Status regs,
				hold_data := BlockStatus((adder - (adder mod 65536))/65536);
									-- put the block status register in
									-- the holding area
				FOR j IN 15 DOWNTO 2 LOOP		
					hold_data(j) := 'X';		-- And X bits 15-2 - they're undefined
				END LOOP;

			ELSIF (adder mod 65536) = 3 THEN
				hold_data := "XXXXXXXXXXXXXXXX";
				IF MstrLockBit = '1' THEN hold_data(0) := '1';
							ELSE hold_data(0) := '0';
				END IF;


			ELSE	hold_data := QueryTable(adder);		-- Otherwise, read from the CFI Query
									-- database

			END IF;                                        

			IF (byteb = '0') THEN				-- If we're in byte mode, turn off
				FOR j IN 15 downto 8 LOOP		-- the top byte of the holding area
					hold_data(j) := 'Z';		-- Again - there's nothing of
				END LOOP;				-- interest there.
			END IF;

		-- END ReadQuery subcase


		WHEN OTHERS =>						-- If we get here, something went
			ASSERT (0>1)					-- dramatically, horribly wrong.
			REPORT "Model failure - unknown ReadType"	-- This simply can't happen.
			SEVERITY FAILURE;

		END CASE;						-- WE HAVE NOW LEFT THE ReadType
									-- CASE STATEMENT!


		IF (byteb'LAST_EVENT < tFLQV) THEN			-- If the byte line has transitioned
									-- in the last tFLQV, the data is
									-- invalid.  This trumps whatever
			hold_data := "XXXXXXXXXXXXXXXX";		-- might cause the data to be valid.

			IF ((byteb = '0') AND (byteb'LAST_EVENT >= tFLQZ)) THEN	-- However, after tFLQZ of
									-- the transition, the top half of
									-- the outputs will shut off.
				hold_data := "ZZZZZZZZXXXXXXXX";	-- Note that this occurs only when
									-- switching TO byte mode
			END IF;
		END IF;


		FOR j IN 15 DOWNTO 0 LOOP				-- Put the stuff in the holding area
			dq(j) <= hold_data(j);				-- onto the data outputs.
		END LOOP;


		equal := '1';						-- This is generic code to see if
		FOR j IN MaxAddrLine downto 0 LOOP			-- the address on the bus has changed
			IF (hold_add(j) /= addr(j)) THEN		-- from the one we latched when we 
			equal := '0';					-- left ReadCycleX.  If it has, 
			END IF;						-- equal will be 0.
		END LOOP;      

		IF (rpbsig = '0') THEN					-- RP is asserted.
			TimeRPL := NOW;					-- Note when RP went low,
			next_state := PowerDown;			-- and go to PowerDown.

		ELSIF ((oeb = '1' AND oeb'LAST_EVENT >= tOH) OR		-- If one of the control lines has
			 (ceb = '1' AND ceb'LAST_EVENT >= tOH) OR	-- transitioned, or the address has
			 (equal = '0' AND addr'LAST_EVENT >= tOH)) THEN -- changed, the data goes invalid
		next_state := ReadCycleX;				-- and we go to ReadCycleX
 
		ELSE next_state := ReadCycleV;				-- Otherwise, stay here
		END IF;        
		
	-- END STATE ReadCycleV




	WHEN WriteCycle =>	-- Unlike the read process, which goes through many substates, the
				-- write process only uses one monolithic state.  To be here is to have
				-- WE and CE low/asserted, and OE and RP high/unasserted.  Even then, the
				-- only interesting things happen on the rising edge of the controlling
				-- signal (usually WE, but not always.) 


		IF (web = '1') THEN  					-- Rising edge, and by implication,
									-- a WE-controlled write.

			next_state := OutputDisable;			-- Go to OutputDisable next

			timeWH := NOW;					-- Note when WE went high

			ASSERT (((NOW - web'LAST_EVENT) - timeWL) >= tWLWH)	
			REPORT "WE Pulse Width violation (tWLWH)."	-- Make sure WE was low long enough
			SEVERITY ERROR;					-- WE Pulse Width

			ASSERT (dq'LAST_EVENT >= tDVWH)			-- Check Data Setup time
			REPORT "Data setup time violation (tDVWH)."
			SEVERITY ERROR;

			ASSERT (addr'LAST_EVENT >= tAVWH)		-- Check Address Setup time
			REPORT "Address setup time violation (tAVWH)."
			SEVERITY ERROR;



		ELSIF (ceb = '1') THEN					-- Rising edge on CE - by implication
									-- a CE-controlled write

			next_state := WEActive;				-- Go to WEActive next

			timeEH := NOW;					-- Note when CE went high

			ASSERT (((NOW - ceb'LAST_EVENT) - timeEL) >= tELEH)
			REPORT "CE Pulse Width violation (tELEH)."	-- Make sure CE was low long enough
			SEVERITY ERROR;					-- CE Pulse Width

			ASSERT (dq'LAST_EVENT >= tDVEH)			-- Check Data Setup time
			REPORT "Data setup time violation (tDVEH)."
			SEVERITY ERROR;

			ASSERT (addr'LAST_EVENT >= tAVEH)		-- Check Address Setup time
			REPORT "Address setup time violation (tAVEH)."
			SEVERITY ERROR;

		END IF;							-- End the if about control lines
									-- going back to high

		IF (rpbsig = '0') THEN					-- RP is asserted.
			next_state := PowerDown;			-- Go to Powerdown
			TimeRPL := NOW;					-- and note when RP went low


		ELSIF (web = '1' OR ceb = '1') THEN			-- If either of the control lines
									-- went high, we do all sorts of stuff

			FOR j IN 0 to MaxAddrLine LOOP			-- Latch the address when a line goes
				hold_add(j) := addr(j);			-- high
			END LOOP;
	
			FOR j IN 0 to 15 LOOP				-- Latch the data when a line goes
				hold_data(j) := dq(j);			-- high
			END LOOP;
	

			adder := 0;					-- Stock code to integerize
			twosum := 1;					-- the address (word address)
			FOR j IN 1 to MaxAddrLine LOOP			-- Is stored in adder
				IF (hold_add(j) = '1') THEN
					adder := adder + twosum;
				END IF;
				twosum := twosum * 2;
			END LOOP;


			byteadder := 0;					-- Integerize the BYTE address, 
			twosum := 1;					-- and store it in byteadder

			FOR j IN 0 TO MaxAddrLine LOOP
				IF (hold_add(j) = '1') THEN
					byteadder := byteadder + twosum;
				END IF;
				twosum := twosum * 2;
			END LOOP;



			data := 0;					-- Integerize the data, and store
			datalo := 0;					-- it in data, but also store the
			twosum := 1;					-- low byte's integer value in datalo

			FOR j IN 0 to 15 LOOP
				IF (hold_data(j) = '1') THEN
					data := data + twosum;
					IF (j<8) THEN 
					datalo := datalo + twosum;
					END IF;
				END IF;
				twosum := twosum * 2;
			END LOOP;


			BlockNum := 0;					-- And last, figure the number of the
			twosum := 1;					-- block to which the latched address
			FOR j IN 17 TO MaxAddrLine LOOP			-- corresponds, and store it in 
				IF hold_add(j) = '1' THEN		-- BlockNum
					BlockNum := BlockNum + twosum;
				END IF;
				twosum := twosum * 2;
			END LOOP;



			CASE WriteType IS				-- The idea here's similar to the 
									-- idea with the ReadType subcase.
									-- WriteType stores what sort of data
									-- the device expects to see on a 
									-- write cycle.


			WHEN WriteCmd =>				-- Case 1 - The data's a command.

				CASE datalo IS				-- On a command, the low byte holds
									-- the important stuff.
							

				WHEN ClearSRCmd =>			-- Got the command to clear the SR.
					IF ((running = 0) AND 		-- If the device isn't busy, and
									-- there isn't anything suspended,
						 (SR(6) = '0' AND SR(2) = '0')) THEN  

						SR := SR AND "11000101"; --Clear the status register.
	
					ELSE  ASSERT(0>1)		-- Otherwise, protest vigorously 
						REPORT 			-- that the command wasn't valid.
						"Invalid Command While Device Busy (ClearSR)"
						SEVERITY ERROR;
					END IF;

				WHEN EraseSingleBlockCmd =>		-- Command to erase one block
					IF ((running = 0) AND 		-- If the device isn't busy, and
									-- there isn't anything suspended,
						 (SR(6) = '0' AND SR(2) = '0')) THEN  
						WriteType := WriteBlkErase;
									-- Go to WriteBlkErase mode
									-- and wait for confirmation.

					ELSE  ASSERT(0>1)		-- Otherwise, the command wasn't good
						REPORT 
						"Invalid Command While Device Busy (EraseBlk)"
						SEVERITY ERROR;
					END IF;

				WHEN ProgramCmd =>			-- Command to program a byte/word.
									-- If the device isn't busy, and
					IF ((running = 0) AND (SR(2) = '0')) THEN  -- no erase is suspended,
						WriteType := WriteProgram;	-- Go to WriteProgram mode,
									-- and wait for the data byte/word.

					ELSE  ASSERT(0>1)		-- Otherwise... yeah.
						REPORT 
						"Invalid Command While Device Busy (Program)"
						SEVERITY ERROR;
					END IF;
	
				WHEN Program2Cmd =>			-- Ditto the above - this is the
									-- alternate value for the same thing
					IF ((running = 0) AND (SR(2) = '0')) THEN  
						WriteType := WriteProgram;
					ELSE  ASSERT(0>1)
						REPORT 
						"Invalid Command While Device Busy (Program)"
						SEVERITY ERROR;
					END IF;

				WHEN ReadArrayCmd =>			-- Command to change the read mode
									-- to read the memory array.
					IF (running = 0) THEN		-- If nothing's running
						ReadType := ReadArray;	-- Change the read mode
					ELSE  ASSERT(0>1)		-- Otherwise, no dice.
						REPORT
						"Invalid Command While Device Busy (ReadArray)"
						SEVERITY ERROR;
					END IF;

				WHEN ReadCSRCmd =>			-- Command to change the read mode
									-- to read the Status Register
					IF (running = 0) THEN		-- If nothing's running
						ReadType := ReadStatReg;	-- change the read mode
					ELSE  ASSERT(0>1)		-- Otherwise, complain.
						REPORT
						"Invalid Command While Device Busy (ReadSR)"
						SEVERITY ERROR;
					END IF;

				WHEN ReadIDCmd =>			-- Command to change the read mode
									-- to read the ID Codes
					IF ((running = 0) AND 		-- If nothing's running, and
									-- nothing's suspended,
						 (SR(6) = '0' AND SR(2) = '0')) THEN  
						ReadType := ReadIDCodes;	-- change the read mode.
					ELSE  ASSERT(0>1)		-- Otherwise, gripe.
						REPORT 
					"Invalid Command While Device Busy (ReadIDCodes)"
						SEVERITY ERROR;
					END IF;

				WHEN ResumeCmd =>			-- Command to resume a suspended
									-- operation
					IF (running = 0) THEN		-- If there's nothing running,

						IF (SR(6) = '1') THEN	-- An erase was suspended.
							SR(6) := '0';	-- Clear the Erase Suspended flag
							SR(7) := '0';	-- Clear the Device Ready flag

							ReadType := ReadStatReg;   -- Update the ReadType
									-- to read the Status Register

							running := WriteBlkErase;
									-- Note that a Block Erase is running

							IF (STSMode = "00") THEN sts <= '0' after tWHRL;
									-- If STS is in RY/BY mode, make it
									-- go busy after tWHRL.
							END IF;
							Completion := NOW + W16_BlockErase;
									-- And note when the erase will 
									-- complete.

						ELSE ASSERT (0>1)	-- There's nothing to resume!
									-- The command's invalid.  Complain.
						REPORT "Invalid Attempt To Resume"
						SEVERITY ERROR;
						END IF;

					ELSE  ASSERT (0>1)		-- The device is busy
									-- which makes the command invalid.
						REPORT "Invalid Attempt To Resume"
						SEVERITY ERROR;
					END IF;



				WHEN SuspendCmd =>			-- Command to suspend whatever is
									-- running.

					IF (running = WriteBlkErase) THEN
										-- A block erase is running
						running := SuspensionLatency;	-- Update running to note
										-- that we're waiting for a
										-- suspension

						Completion := NOW + W16_EraseSuspToRead; 
										-- Update the completion time
										-- to the new estimate
						susp_add := hold_add;		-- Save off the address now
										-- in the latch

						SR(7) := '0';			-- Clear the Device Ready
										-- flag

						SR(6) := '1';			-- Set the Erase Suspended
										-- flag

						IF (STSMode = "00") THEN sts <= '0' after tWHRL;
										-- If STS is in RY/BY, make
										-- STS low after tWHRL
						END IF;					

					ELSIF (running = 0) THEN		-- If nothing's running,
						ASSERT(0>1)			-- the command isn't valid.
						REPORT
					"Invalid Command While Device Not Busy (Suspend)"
						SEVERITY ERROR;

					ELSE  ASSERT(0>1)			-- This may never happen.
						REPORT				-- It means the device is
					"Invalid Attempt To Suspend"		-- busy with something that
						SEVERITY ERROR;			-- can't be suspended.
					END IF;					-- Full Chip Erase, perhaps.


				WHEN ReadQueryCmd =>				-- Command to read from the
										-- CFI Query database
					IF ((running = 0) AND 			-- If there's nothing running 
						(SR(6) = '0' AND SR(2) = '0')) THEN  
										-- and nothing suspended,
						ReadType := ReadQuery;		-- then update ReadType

					ELSE  ASSERT(0>1)			-- Otherwise, complain
						REPORT 
					"Invalid Command While Device Busy (ReadQuery)"
						SEVERITY ERROR;
					END IF;


				WHEN WriteBufferCmd =>		
					IF (SR(6) = '0' AND SR(2) = '0') THEN  
						ReadType := ReadXStatReg;
						IF (XSR(7) = '1') THEN
							WriteType := WriteBufNum;
						ELSE	WriteType := WriteCmd;
						END IF;
					ELSE  ASSERT(0>1)
						REPORT 
					"Invalid Command While Device Busy (WriteToBuffer)"
						SEVERITY ERROR;
					END IF;

				WHEN STSConfigCmd =>				-- Command to reconfigure
										-- the STS pin
					IF ((running = 0) AND 			-- If nothing's running,
						 (SR(6) = '0' AND SR(2) = '0')) THEN  
										-- and nothing's suspended,
						WriteType := WriteSTSPinCfg;	-- then change modes to
										-- wait for the new pin mode

					ELSE  ASSERT(0>1)			-- Otherwise... 
						REPORT 				-- you know the drill.
					"Invalid Command While Device Busy (WriteSTSPinCfg)"
						SEVERITY ERROR;
					END IF;

				WHEN SetBlkLockbitCmd =>			-- Could be command to set a
										-- block lockbit, or the cmd
										-- to clear all the lockbits

					IF ((running = 0) AND			-- If nothing's running

						 (SR(2) = '0' AND SR(6) = '0')) THEN  
										-- And nothing's suspended
						WriteType := WriteLockBit;	-- Then wait for data

					ELSE  ASSERT(0>1)			-- Otherwise, it's an 
						REPORT				-- invalid command
					"Invalid Command While Device Busy (SetLockBit)"
						SEVERITY ERROR;
					END IF;

					

				WHEN OTHERS =>					-- This is an error condition
					ASSERT (0>1)				-- corresponding to some
					REPORT "Unrecognized command written."	-- command being sent which 
					SEVERITY ERROR;				-- isn't actually a command.
										-- Don't yet know what the 
				END CASE;					-- device really does here.




			WHEN WriteProgram =>					-- This subcase means that 
										-- the device is expecting
										-- the second byte of the
										-- Program a Byte/Word cmd.

				ReadType := ReadStatReg;			-- A read gives the SR now

				equal := '0';					-- This loop establishes
										-- if the attempted program
				IF (SR(6) = '1') THEN				-- means to write to a byte
					equal := '1';				-- or word which is part of
					FOR J IN 17 TO MaxAddrLine LOOP		-- a suspended erase.
						IF (hold_add(j) /= susp_add(j)) THEN
							equal := '0';
						END IF;
					END LOOP;
				END IF;

				-- Check versus block being erased		-- If equal = 1, then the
				IF (equal = '1') THEN				-- address is part of an 
					SR := SR OR "00010000";			-- erase, and the prog fails
				END IF;
	
				-- Check versus lockbits
				IF (BlockStatus(BlockNum)(0) = '1' AND rpbhighsig = '0') THEN
					SR := SR OR "00010010";			-- If WP is asserted, and the
				END IF;						-- lock bit for this block is
										-- set, then the prog fails

				-- Check versus powerdown
				IF (rpbsig = '0') THEN SR := SR OR "00010000";	-- If we have entered
				END IF;						-- powerdown, the prog fails

				-- Check Vcc

				IF (VccSig = '0') THEN SR := SR OR "00010000";	-- If Vcc is out of range,
				END IF;						-- the prog fails

				-- Check Vpp

				IF (vpen < Vpp5vLockout) THEN			-- If Vpp is below the
					SR := SR OR "00011000";			-- lockout voltage, the 
				END IF;						-- prog fails

				IF (VppSig = '0') THEN SR := SR OR "00011000";	-- If Vpp is out of range, 
				END IF;						-- the prog fails


				IF NOT ((equal = '1') OR (BlockStatus(BlockNum)(0) = '1'
				   AND rpbhighsig = '0')
				   OR (rpbsig = '0') OR (VccSig = '0') OR (VppSig = '0')) THEN
										-- If nothing's gone wrong,
										-- program the memory.
					IF (web = '1') THEN
						ASSERT (VppSig'LAST_EVENT >= tVPWH)
						REPORT "Violation of Vpp setup time (tVPWH)"
						SEVERITY ERROR;
										-- If it's a WE-controlled
										-- write, verify that Vpp was
										-- high early enough
										-- Vpp Setup time

					ELSIF (ceb = '1') THEN
						ASSERT (VppSig'LAST_EVENT >= tVPEH)
						REPORT "Violation of Vpp setup time (tVPEH)"
						SEVERITY ERROR;
										-- If it's a CE-controlled
										-- write, verify that Vcc was
										-- high early enough
										-- Vcc Setup time

					END IF;


					ProgrammingAddr := adder;

					IF (byteb = '1') THEN			-- If we're in word mode,
								
						equal := '1';			-- This loop serves to check
										-- if the memory can be 
										-- programmed with the
										-- intended value, or if an
										-- erase would be needed

						FOR j IN 0 TO 15 LOOP
							IF ((hold_data(j) AND (NOT(memory(adder)(j)))) = '1') THEN
								equal := '0';
							END IF;
						END LOOP;

						IF (equal = '0')		-- Equal = 0 means that the
										-- value couldn't be
										-- programmed as desired 
										-- without first erasing the
										-- cell. 
						THEN 	SR := SR OR "00010000";	
										-- Set the prog fail flag

						ELSE 	memory(adder) := hold_data;
										-- Otherwise, write the value
										-- to the memory array.
						END IF;
				





					ELSE 	equal := '1';			-- We're in byte mode, which
										-- is far more difficult to 
										-- handle.

						IF (hold_add(0) = '0') THEN	-- If the address is for the
										-- low byte,

										-- Do the same sort of loop
										-- to check if the value can
										-- be programmed without an
										-- erase
							FOR j IN 0 TO 7 LOOP
								IF ((hold_data(j) AND 
									(NOT(memory(adder)(j)))) = '1')
									THEN equal := '0'; 
								END IF;
							END LOOP;

										-- Otherwise, address is for
										-- the high byte.

										-- Same loop, but over the 
										-- other half of the word.

						ELSE	FOR j IN 0 TO 7 LOOP
								IF ((hold_data(j) AND 
									(NOT(memory(adder)(j+8)))) = '1')
									THEN equal := '0';
								END IF;
							END LOOP;
						END IF;



						IF (equal = '1') THEN		-- If equal = 1, then the
										-- device can be programmed.

							IF (hold_add(0) = '0') THEN	-- Program the low
								FOR j IN 0 TO 7 LOOP	-- byte...

									memory(adder)(j) := hold_data(j);
								END LOOP;


							ELSE	FOR j IN 0 TO 7 LOOP	-- Program the high
											-- byte...

									memory(adder)(j+8) := hold_data(j);
								END LOOP;
							END IF;
										-- Otherwise, the device 
										-- couldn't be programed.
						ELSE SR := SR OR "00010000";	-- Set the prog fail flag.
						END IF;
					END IF;

					running := WriteProgram;		-- Note that a program op
										-- is what's making the dev
										-- busy.
					SR(7) := '0';				-- Clear the Dev Ready flag

										-- If STS is in RY/BY mode,
										-- make it busy after tWHRL

					IF (STSMode = "00") THEN sts <= '0' after tWHRL; 
					END IF;

										-- Update the expected time
										-- that the busyness will end
					IF (byteb = '0') THEN			 
						Completion := NOW + W16_ProgByteWOBuffer;           
					ELSE	Completion := NOW + W16_ProgWordWOBuffer;
					END IF;

				END IF;  -- The one starting with (equal ?= '1')       

				WriteType := WriteCmd;				-- And after all that, the
										-- next thing written is a
										-- command.
			

			WHEN WriteBlkErase =>					-- Command to erase one blk

				ReadType := ReadStatReg;			-- Next read shall be reading
										-- the status register.

				IF (datalo /= ConfirmCmd) THEN			-- If the second byte wasn't
					SR := SR OR "00110000";			-- a CONFIRM, set the invalid
				END IF;						-- command flags


				-- Check versus lockbits			-- If WP's asserted and the
										-- block is locked, the erase
				IF (BlockStatus(BlockNum)(0) = '1' 
				AND rpbhighsig = '0') THEN
					SR := SR OR "00100010";			-- fails - set flags
				END IF;						-- accordingly

				-- Check versus powerdown			-- If RP's asserted, the
				IF (rpbsig = '0') THEN SR := SR OR "00100000";	-- device should be powering
				END IF;						-- down, not erasing stuff.
										-- The erase fails.
										-- Set Erase Fail flag.

				-- Check Vcc					-- If Vcc is out of range,
				IF (VccSig = '0') THEN SR := SR OR "00100000";	-- the erase certainly fails.
				END IF;						-- Set Erase Fail flag.

				-- Check Vpp					-- If Vpp is below lockout,
				IF (vpen < Vpp5vLockout) THEN			-- the erase fails.
					SR := SR OR "00101000";			-- Set Erase Fail flag
				END IF;						-- and Vpp Low flag

				IF (VppSig = '0') THEN SR := SR OR "00101000";	-- If Vpp is out of range,
				END IF;						-- the erase fails.
										-- Set Erase Fail flag
										-- and Vpp Low flag


				IF NOT ((datalo /= ConfirmCmd) OR  
					((BlockStatus(BlockNum)(0) = '1')
					AND rpbhighsig = '0') OR		-- If none of those things 
					(rpbsig = '0') OR (VccSig = '0') OR	-- happened, then erase the
					(VppSig = '0')) THEN			-- block


					IF (web = '1') THEN			-- If the write was
										-- WE-controlled, check that
										-- Vpp was high early enough.
										-- Vpp Setup time
						ASSERT (VppSig'LAST_EVENT >= tVPWH)
						REPORT "Violation of Vpp setup time (tVPWH)"
						SEVERITY ERROR;


					ELSIF (ceb = '1') THEN			-- If the write was
										-- CE-controlled, check that
										-- Vpp was high early enough.
										-- Vpp Setup Time
						ASSERT (VppSig'LAST_EVENT >= tVPEH)
						REPORT "Violation of Vpp setup time (tVPEH)"
						SEVERITY ERROR;
					END IF;

					ErasingBlk := BlockNum;

										-- Here, we write all 1's to
					FOR j IN 0 TO 65535 LOOP		-- the block, which erases it
					memory((BlockNum * 65536) + j) := "1111111111111111";
					END LOOP;
				
					running := WriteBlkErase;		-- Mark that a block erase is
										-- busying the device

					SR(7) := '0';				-- Clear the Dev Ready flag

										-- If STS is in RY/BY mode,
										-- make it busy after tWHRL
					IF (STSMode = "00") THEN sts <= '0' after tWHRL;
					END IF;
					Completion := NOW + W16_BlockErase;	-- Note when we expect to
										-- finish the erase.
					
				END IF;

				WriteType := WriteCmd;				-- And the next write will be
										-- a command.



			WHEN WriteBufNum =>

				ActiveBuffer := 0;
				FOR j IN 1 TO NumBuffers LOOP
					IF ((ActiveBuffer = 0) AND (BufStatus(j) = BufFree)) THEN
						ActiveBuffer := j;
					END IF;
				END LOOP;
				IF (ActiveBuffer = 0) THEN XSR(7) := '0';
				END IF;
 
				Bufbyteblatch(ActiveBuffer) := byteb;

				ReadType := ReadStatReg; 
			
				IF (byteb = '0') THEN
					IF datalo > ByteMaxBufWrite THEN
						ASSERT (0>1)
						REPORT "Buffer write fails - too large for buffer."
						SEVERITY ERROR;
						-- generate some error - too big a write
						WriteType := WriteCmd;
					ELSE
						BufBlockNum(ActiveBuffer) := BlockNum;
						BufCount(ActiveBuffer) := datalo;
						BufStartAddr(ActiveBuffer) := byteadder;
						BufEndAddr(ActiveBuffer) := (byteadder + datalo);
						WriteType := WriteBufData;
					END IF;
				ELSE	IF data > WordMaxBufWrite THEN
						ASSERT (0>1)
						REPORT "Buffer write fails - too large for buffer."
						SEVERITY ERROR;

						-- generate some error - too big a write
						WriteType := WriteCmd;
					ELSE 
						BufBlockNum(ActiveBuffer) := BlockNum;
						BufCount(ActiveBuffer) := data;
						BufStartAddr(ActiveBuffer) := adder;
						BufEndAddr(ActiveBuffer) := (adder + data);
						WriteType := WriteBufData;
					END IF;			
				END IF;
				countdown := BufCount(ActiveBuffer);
				BufStatus(ActiveBuffer) := BufReading;


			WHEN WriteBufData =>

				ActiveBuffer := 0;
				FOR j IN 1 TO NumBuffers LOOP
					IF BufStatus(j) = BufReading THEN ActiveBuffer := j;
					END IF;
				END LOOP;


				IF BlockNum /= BufBlockNum(ActiveBuffer) THEN
					SR := SR OR "00110000";
					WriteType := WriteCmd;
					BufStatus(ActiveBuffer) := BufFree;
				END IF;

				IF BufbytebLatch(ActiveBuffer) = '0' THEN
					IF (byteadder < BufStartAddr(ActiveBuffer)) OR
					   (byteadder > BufEndAddr(ActiveBuffer)) THEN

						ASSERT (0>1)
						REPORT "Buffer write fails - datum out of range."
						SEVERITY ERROR;

						-- Throw some error - write out of range.
						BufStatus(ActiveBuffer) := BufFree;
						WriteType := WriteCmd;	-- This may not be right

					ELSE	IF (byteadder mod 2 = 0) AND (BufStartAddr(ActiveBuffer) mod 2 = 0)
						THEN	FOR j IN 0 TO 7 LOOP
							WriteBuffers(ActiveBuffer,
								     (byteadder - BufStartAddr(ActiveBuffer)) / 2)
								     (j) := hold_data(j);
							END LOOP;
						ELSIF (byteadder mod 2 /= 0) AND
						      (BufStartAddr(ActiveBuffer) mod 2 = 0)
						THEN	FOR j IN 0 TO 7 LOOP
							WriteBuffers(ActiveBuffer, (byteadder - 1 - BufStartAddr
	 							     (ActiveBuffer)) / 2)(j+8) := hold_data(j);
							END LOOP;
						ELSIF (byteadder mod 2 = 0) THEN
							FOR j IN 0 TO 7 LOOP
							WriteBuffers(ActiveBuffer, (byteadder + 1 - BufStartAddr
								     (ActiveBuffer)) / 2)(j) := hold_data(j);
							END LOOP;
						ELSE	FOR j IN 0 TO 7 LOOP
							WriteBuffers(ActiveBuffer, (byteadder -
								      BufStartAddr(ActiveBuffer)) / 2)(j+8)
								      := hold_data(j);
							END LOOP;
						END IF;
					END IF;


				ELSE	IF (adder < BufStartAddr(ActiveBuffer)) OR
					   (adder > BufEndAddr(ActiveBuffer)) THEN
						-- Throw some error - write out of range (as above).
						ASSERT (0>1)
						REPORT "Buffer write fails - datum out of range."
						SEVERITY ERROR;

						WriteType := WriteCmd;	--??????
						BufStatus(ActiveBuffer) := BufFree;

					ELSE	WriteBuffers(ActiveBuffer, (adder - BufStartAddr(ActiveBuffer)))
							:= hold_data;
					END IF;
				END IF;

				

				IF WriteType /= WriteCmd THEN
					IF countdown = 0 THEN
						WriteType := WriteBufCfrm;
					ELSE	WriteType := WriteBufData;
						countdown := countdown - 1;
					END IF;
				END IF;
					


			WHEN WriteBufCfrm =>
				
				ActiveBuffer := 0;			-- Find which buffer just filled
				FOR j IN 1 TO NumBuffers LOOP
					IF BufStatus(j) = BufReading THEN ActiveBuffer := j;
					END IF;
				END LOOP;


				IF (datalo /= ConfirmCmd) THEN 		-- If the write isn't confirmed,
					SR := SR OR "00110000";		-- set invalid cmd flags
				END IF;


				-- Check versus lockbits		-- If WP's asserted and the 
									-- lockbit is set, 
				IF (BlockStatus(BlockNum)(0) = '1' AND rpbhighsig = '0') THEN
					SR := SR OR "00100010";		-- Set flags accordingly
				END IF;

				-- Check versus powerdown		-- If RP's asserted
				IF (rpbsig = '0') THEN SR := SR OR "00100000";	-- the write fails
				END IF;

				-- Check Vcc				-- If Vcc's out of range, 
				IF (VccSig = '0') THEN SR := SR OR "00100000";	-- the write fails
				END IF;

				-- Check Vpp				-- If Vpp's below lockout,
				IF (vpen < Vpp5vLockout) THEN			-- the write fails
					SR := SR OR "00101000";
				END IF;

				IF (VppSig = '0') THEN SR := SR OR "00101000";	-- If Vpp's out of range
				END IF;						-- the write fails

										
				IF NOT ((datalo /= ConfirmCmd) OR 		-- If all's well
					((BlockStatus(BlockNum)(0) = '1') AND rpbhighsig = '0')
					OR (rpbsig = '0') OR (VccSig = '0') OR (VppSig = '0')) THEN

					BufStatus(ActiveBuffer) := BufFull;	-- mark the buffer as full
										-- which implies queued to
										-- write to the flash

										-- and if there are any free
										-- write buffers left. 
					XSR(7) := '0';
					FOR j IN 1 TO NumBuffers LOOP
						IF BufStatus(j) = BufFree THEN XSR(7) := '1';
						END IF;
					END LOOP;






					IF (running = 0) THEN			-- If nothing else is going
										-- on, go ahead and let it
										-- start writing to the flash

						IF (BufbytebLatch(ActiveBuffer) = '1') THEN
										
										-- If it was all done in 
										-- word mode...

							FOR j IN 0 TO BufCount(ActiveBuffer) LOOP

										-- Check if each program can
										-- be done without having
										-- to erase the memory.
										-- We cheat, though - the 
										-- loop always says you can't
										-- program if there's already
										-- a program fail in the SR.

								equal := '0';

								IF SR(4) = '0' THEN equal := '1';	
								END IF;

								FOR k IN 0 TO 15 LOOP
									IF ((WriteBuffers(ActiveBuffer,j)(k) AND 
									(NOT(memory(BufStartAddr
									(ActiveBuffer) + j)(k)))) 
								 	= '1') THEN equal := '0';
									END IF;
								END LOOP;

										-- If we can program, and
										-- nothing failed already, 
										-- do program.

								IF (equal = '1')
								THEN 	memory((BufStartAddr(ActiveBuffer) -
										1) /2 + j) :=
									WriteBuffers(ActiveBuffer, j);

										-- If something failed,
										-- set the prog fail flag.

								ELSE 	SR(4) := '1';
								END IF;
							END LOOP;

										-- Note the buffer is writing	
										-- to the flash,
							BufStatus(ActiveBuffer) := BufWriting;

										-- that a buffer write is
										-- why the device is busy,
							running := WriteType;

										-- that the next read will 
										-- be of the SR,
							ReadType := ReadStatReg;

										-- when we expect to finish,
							Completion := NOW + ((BufCount(ActiveBuffer) + 1) 
										* W16_ProgWordWBuffer);
						
										-- that the device is not
										-- ready,
							SR(7) := '0';

							IF STSMode = "00" THEN
								sts <= '0' after tWHRL;
							END IF;



										-- Otherwise, the write was
										-- started in byte mode, and
										-- things get much more
										-- complicated.

						-- The byte mode version of the write to flash break into
						-- three parts.

						-- 1) If the starting address is odd, the write starts by 
						--    writing to only the high byte of the first address.
						-- 2) The middle portion of the write goes a word at a time.
						-- 3) If the ending address is even, the write ends by
						--    writing to only the low byte of the ending address.


										-- This is the first part.
						ELSE	IF (BufStartAddr(ActiveBuffer) mod 2 = 1) THEN

										-- This loop checks both
										-- if any program errors
										-- have occurred and if the
										-- current byte can be 
										-- programmed without erasing

								equal := '1';
								IF SR(4) = '1' THEN equal := '0';
								END IF;

								FOR k IN 8 TO 15 LOOP
									IF (WriteBuffers(ActiveBuffer, 0)(k) AND
										NOT(memory((BufStartAddr
										(activeBuffer) - 1) / 2)(k)))
										 = '1' THEN equal := '0';
									END IF;
								END LOOP;

										-- And this loop programs the
										-- high byte

								IF equal = '1' THEN
									FOR k IN 8 TO 15 LOOP
										memory((BufStartAddr
										(ActiveBuffer)-1)/2 )(k) := 
										WriteBuffers(ActiveBuffer, 0)(k);
									END LOOP;
								ELSE	SR(4) := '1';
								END IF;
								j_start := 1;	-- This notes where in the
										-- buffer to begin the middle
										-- portion of the write

							ELSE	j_start := 0;	-- So does this.
							END IF;

										-- This logic notes at what
										-- word of the buffer to end
										-- the middle portion of the
										-- write.
							
							IF BufEndAddr(ActiveBuffer) mod 2 = 0 THEN
								IF j_start = 1 THEN
									j_end := (BufCount(ActiveBuffer) + 1) 
											/ 2 - 1;
								ELSE 	j_end := (BufCount(ActiveBuffer) 
											/ 2) - 1;
								END IF;
							ELSE	IF j_start = 1 THEN
									j_end := BufCount(ActiveBuffer)  
											/ 2;
								ELSE	j_end := (BufCount(ActiveBuffer) + 1) 
											/ 2 - 1;
								END IF;
							END IF;


										-- This performs the middle
										-- portion of the write.
							FOR j IN j_start TO j_end LOOP


										-- This loop checks both
										-- if any program errors
										-- have occurred and if the
										-- current byte can be 
										-- programmed without erasing
								equal := '1';
								IF SR(4) = '1' THEN equal := '0';
								END IF;

								FOR k IN 0 TO 15 LOOP
									IF (J_start = 1) THEN

									IF (WriteBuffers(ActiveBuffer, j)(k) AND
									NOT(memory((BufStartAddr(ActiveBuffer)
									-1)/2 + j)(k))) = '1' 
									THEN equal := '0';
									END IF;
					
									ELSE

									IF (WriteBuffers(ActiveBuffer, j)(k) AND
									NOT(memory(BufStartAddr(ActiveBuffer)
									/2 + j)(k))) = '1' 
									THEN equal := '0';
									END IF;
								
									END IF;

								END LOOP;

										-- This loop programs the
										-- middle words.
									
								IF equal = '1' THEN
									FOR k IN 0 TO 15 LOOP
										IF (j_start = 0) THEN
										memory(BufStartAddr(ActiveBuffer)
											 / 2 + j)(k) := 
										WriteBuffers(ActiveBuffer, j)(k);
										ELSE 
										memory((BufStartAddr(ActiveBuffer)
											- 1) / 2 + j)(k) :=
										WriteBuffers(ActiveBuffer, j)(k);
										END IF;
									END LOOP;
								ELSE	SR(4) := '1';
								END IF;
							END LOOP;


										
										-- This is the last part of
										-- the write.
							IF (BufEndAddr(ActiveBuffer) mod 2 = 0) THEN


										-- This loop checks both
										-- if any program errors
										-- have occurred and if the
										-- current byte can be 
										-- programmed without erasing
								j_end := j_end + 1;
								equal := '1';
								IF SR(4) = '1' THEN equal := '0';
								END IF;

								FOR k IN 0 TO 7 LOOP

									IF j_start = 0 THEN

									IF (WriteBuffers(ActiveBuffer, j_end)
									(k) AND NOT(memory(BufStartAddr
									(ActiveBuffer)/2 + j_end)(k))) = '1' 
									THEN 	equal := '0';
									END IF;

									ELSE 

									IF (WriteBuffers(ActiveBuffer, j_end)
									(k) AND NOT(memory((BufStartAddr
									(ActiveBuffer)- 1)/2 + j_end)(k))) = '1' 
									THEN 	equal := '0';
									END IF;
	
									END IF;

								END LOOP;

										-- This programs the 
										-- straggler low byte

								IF equal = '1' THEN
									FOR k IN 0 TO 7 LOOP

										IF j_start = 0 THEN
										memory(BufStartAddr(ActiveBuffer)
											/ 2 + j_end)(k) := 
										WriteBuffers(ActiveBuffer, j_end)
											(k);

										ELSE memory((BufStartAddr
										(ActiveBuffer)- 1)/ 2 + j_end)(k) 
											:= 
										WriteBuffers(ActiveBuffer, j_end)
											(k);
										END IF;

									END LOOP;
								ELSE	SR(4) := '1';
								END IF;
						
							END IF;	
						


										-- Note the buffer is writing	
										-- to the flash,
							BufStatus(ActiveBuffer) := BufWriting;

										-- that a buffer write is
										-- why the device is busy,
							running := WriteType;

										-- that the next write will
										-- be a command,
							ReadType := ReadStatReg;

										-- when we expect to finish,
							Completion := NOW + ((BufCount(ActiveBuffer) + 1) 
										* W16_ProgByteWBuffer);
						
										-- that the device is not
										-- ready,
							SR(7) := '0';
							IF STSMode = "00" THEN
								sts <= '0' AFTER tWHRL;
							END IF;

						END IF;
					END IF;
				END IF;
				
				WriteType := WriteCmd;

			-- END Section temporarily without comments.




			WHEN WriteSTSPinCfg =>					-- This command reconfigures
										-- the STS pin.

				IF (datalo > 3) THEN				-- The spec is a little
										-- unclear here.  If the low
										-- byte of the latched data
										-- is > $03, the config code
										-- is invalid (depending on
										-- how one interprets the
										-- spec).

					SR := SR OR "00110000";			-- Set Invalid Cmd flags.
				ELSE						-- Otherwise, the code is
					STSMode(1) := hold_data(1);		-- good, and we set the STS
					STSMode(0) := hold_data(0);		-- Pin config bits accordingly
				END IF;
				WriteType := WriteCmd;				-- Next write is a command.




			WHEN WriteLockBit =>					-- Either a command to set
										-- one lockbit, or a cmd
										-- to clear them all.

				IF (datalo = ConfirmCmd) THEN			-- If the cmd is to clear 
										-- the lockbits,

					IF (MstrLockBit = '1' AND rpbhighsig = '0') THEN
						SR := SR OR "00100010";		-- If the master lockbit
					END IF;					-- is set and not overriden
										-- the lockbits don't clear

					IF (vpen < Vpp5vLockout) THEN		-- If Vpp < Vpp Lockout
										-- the lockbits don't clear
						SR := SR OR "00101000";		-- Set Erase Fail and 
					END IF;					-- Vpp Low flags

					-- Check versus powerdown		-- If RP is asserted, the
										-- lockbits don't clear.
					IF (rpbsig = '0') THEN SR := SR OR "00100000";	-- Set Erase Fail
					END IF;

					-- Check Vcc				-- If Vcc is out of range,
										-- the lockbits don't clear.
					IF (VccSig = '0') THEN SR := SR OR "00100000";	-- Set Erase Fail
					END IF;

					-- Check Vpp				-- If Vpp is out of range,
										-- the lockbits don't clear.
					IF (VppSig = '0') THEN SR := SR OR "00101000";	-- Set Erase Fail
					END IF;

					IF NOT ((rpbsig = '0') OR 
						(VccSig = '0') OR (VppSig = '0') OR
						(MstrLockBit = '1' AND rpbhighsig = '0')) 
										-- If none of those things
										-- happened, then...

						THEN
						IF (web = '1') THEN		-- If the write was
										-- WE-controlled, then check
										-- that Vpp was high early
										-- enough - Vpp Setup time
							ASSERT (VppSig'LAST_EVENT >= tVPWH)
							REPORT "Violation of Vpp setup time (tVPWH)"
							SEVERITY ERROR;


						ELSIF (ceb = '1') THEN		-- If the write was
										-- CE-controlled, then check
										-- that Vpp was high early
										-- enough - Vpp Setup time
							ASSERT (VppSig'LAST_EVENT >= tVPEH)
							REPORT "Violation of Vpp setup time (tVPEH)"
							SEVERITY ERROR;
						END IF;
						
										-- Erase all the lockbits
						FOR j IN 0 TO (2**(MaxAddrLine - 16)) LOOP
							BlockStatus(j)(0) := '0';
						END LOOP;

										-- A set/clear lockbit op
						running := WriteLockBit;	-- is what's busying the dev

						SR(7) := '0';			-- Clear Device Ready flag

										-- If the STS pin is in
										-- RY/BY mode, make it busy
						IF (STSMode = "00") THEN sts <= '0' after tWHRL;
						END IF;
										-- Write down when we expect
										-- to be unbusy.
						Completion := NOW + W16_ClearLockBit;
					END IF;					-- End of the Erase Lockbits
										-- routine


				ELSIF (datalo = SetBlkLockBitCmd2) THEN		-- If the command is to set
										-- a lockbit...

					IF (MstrLockBit = '1' AND rpbhighsig = '0') THEN
						SR := SR OR "00010010";		-- If the master lockbit
					END IF;					-- is set and not overriden
										-- the lockbits doesn't set



					IF (vpen < Vpp5vLockout) THEN		-- If Vpp is below lockout,
										-- the set fails.
						SR := SR OR "00011000";		-- Set Prog Fail and Vpp Low
					END IF;					-- flags

					-- Check versus powerdown		-- If RP is asserted, the
					IF (rpbsig = '0') THEN			-- set fails. 
						SR := SR OR "00010000";		-- Set Prog Fail flag
					END IF;

					-- Check Vcc				-- If Vcc is out of range,
					IF (VccSig = '0') THEN 			-- the set fails.
						SR := SR OR "00010000";		-- Set Prog Fail flag
					END IF;

					-- Check Vpp				-- If Vpp is out of range,
					IF (VppSig = '0') THEN SR := SR OR "00011000";	-- the set fails.
										-- Set Prog Fail and Vpp Low
					END IF;					-- flags

					IF NOT ((rpbsig = '0') 
					OR (VccSig = '0') OR (VppSig = '0')	-- If none of those things
					OR (MstrLockBit = '1' AND rpbhighsig = '0')) 	-- happened,
					THEN	


						IF (web = '1') THEN		-- WE-controlled write?
										-- Verify Vpp was high
										-- early enough - 
										-- Vpp Setup Time

							ASSERT (VppSig'LAST_EVENT >= tVPWH)
							REPORT "Violation of Vpp setup time (tVPWH)"
							SEVERITY ERROR;


						ELSIF (ceb = '1') THEN		-- CE-controlled write?
										-- Verify Vpp was high
										-- early enough - 
										-- Vpp Setup Time
							ASSERT (VppSig'LAST_EVENT >= tVPEH)
							REPORT "Violation of Vpp setup time (tVPEH)"
							SEVERITY ERROR;
						END IF;


						BlockNum := 0;			-- Stock code to find the
						twosum := 1;			-- block number of the
						FOR j IN 17 TO MaxAddrLine LOOP	-- addr in the latch
							IF hold_add(j) = '1' THEN
							BlockNum := BlockNum + twosum;
							END IF;
							twosum := twosum * 2;
						END LOOP;
										-- And set that block's 
						BlockStatus(BlockNum)(0) := '1';	-- lockbit.
		
						running := WriteLockBit;	-- Busy due to a lockbit op
						SR(7) := '0';			-- Clr Device Ready flag

										-- If STS is in RY/BY mode
										-- mark it busy
						IF (STSMode = "00") THEN sts <= '0' after tWHRL;
						END IF;
										-- Note when we expect to
										-- be finished.
						Completion := NOW + W16_SetLockBit;
					END IF;        

				ELSIF datalo = SetMstrLockBitCmd2 THEN

					IF rpbhighsig = '0' THEN
						SR := SR OR "00010010";		-- If rpb is not very high
					END IF;					-- the master lock doesn't
										-- set

					IF (vpen < Vpp5vLockout) THEN		-- If Vpp is below lockout,
										-- the set fails.
						SR := SR OR "00011000";		-- Set Prog Fail and Vpp Low
					END IF;					-- flags

					-- Check versus powerdown		-- If RP is asserted, the
					IF (rpbsig = '0') THEN			-- set fails. 
						SR := SR OR "00010000";		-- Set Prog Fail flag
					END IF;

					-- Check Vcc				-- If Vcc is out of range,
					IF (VccSig = '0') THEN 			-- the set fails.
						SR := SR OR "00010000";		-- Set Prog Fail flag
					END IF;

					-- Check Vpp				-- If Vpp is out of range,
					IF (VppSig = '0') THEN SR := SR OR "00011000";	-- the set fails.
										-- Set Prog Fail and Vpp Low
					END IF;					-- flags

					IF NOT ((rpbsig = '0') 
					OR (VccSig = '0') OR (VppSig = '0')	-- If none of those things
					OR rpbhighsig = '0') 			-- happened,
					THEN	


						IF (web = '1') THEN		-- WE-controlled write?
										-- Verify Vpp was high
										-- early enough - 
										-- Vpp Setup Time

							ASSERT (VppSig'LAST_EVENT >= tVPWH)
							REPORT "Violation of Vpp setup time (tVPWH)"
							SEVERITY ERROR;


						ELSIF (ceb = '1') THEN		-- CE-controlled write?
										-- Verify Vpp was high
										-- early enough - 
										-- Vpp Setup Time
							ASSERT (VppSig'LAST_EVENT >= tVPEH)
							REPORT "Violation of Vpp setup time (tVPEH)"
							SEVERITY ERROR;
						END IF;


						MstrLockBit := '1';		-- Sets the master lockbit
		
						running := WriteLockBit;	-- Busy due to a lockbit op
						SR(7) := '0';			-- Clr Device Ready flag

										-- If STS is in RY/BY mode
										-- mark it busy
						IF (STSMode = "00") THEN sts <= '0' after tWHRL;
						END IF;
										-- Note when we expect to
										-- be finished.
						Completion := NOW + W16_SetLockBit;
					END IF;        
					

										-- If it wasn't an erase,
										-- and it wasn't a set,
										-- it's an invalid command.
				ELSE SR := SR OR "00110000";			-- Set flags accordingly.
				END IF; 

				WriteType := WriteCmd;				-- And the next write's a
										-- command.
					

			WHEN OTHERS =>						-- This should never happen
				ASSERT (0>1)					-- If we're here, the
				REPORT "Unrecognized WriteType achieved."	-- WriteType state machine
				SEVERITY FAILURE;				-- left all the valid states

			
			END CASE;						-- Case of write types

										-- This is kinda sloppy
										-- After every write cycle,
										-- we check the memory
										-- location in the latch to
										-- see if we wrote Zs to it
										-- by mistake.  If we did,
			FOR j IN 15 DOWNTO 0 LOOP				-- turn the Zs into Xs.
			IF (memory(adder)(j) = 'Z') THEN memory(adder)(j) := 'X';
				END IF;
			END LOOP;

		END IF;        -- If WE rising edge encountered




	WHEN PowerDown =>	-- This state corresponds to RP asserted, and who cares about the rest of
				-- the lines?  RP asserted means we're resetting and powered down.

		STSMode := "00";						-- STS is set to RY/BY
		IF (running /= 0) AND (running /= ResetLatency) THEN		-- If something's running, 
			running := ResetLatency;				-- it takes longer to reset.
			Completion := NOW + tPLRH;				-- Note we're resetting,
			sts <= '0';						-- how long it will take, and
										-- mark STS busy.
		
		ELSE								-- If nothing's running
			SR := "10000000";					-- clear the status register
			dq <= "ZZZZZZZZZZZZZZZZ";				-- turn off the outputs
			ReadType := ReadArray;					-- and get ready to read from
		END IF;								-- memory.
		
		IF (rpbsig = '1') THEN						-- RP is no longer asserted
			next_state := Powerup;					-- We're going to powerup
			sts <= 'Z';						-- STS - device is free

			ASSERT ((NOW - TimeRPL) >= TPLPH) OR (TimeRPL = 0 ns)	-- Check that RP was asserted
			REPORT "Reset Pulse Width violation (TPLPH)"		-- long enough 
			SEVERITY ERROR;						-- Reset Pulse width

			ASSERT (running = 0)					-- Check that, if something
										-- was running, that RP was
			REPORT "Reset Pulse Width violation (TPLRH)"		-- asserted long enough
			SEVERITY ERROR;						-- Reset Pulse width

			running := 0;						-- If we're leaving powerdown
			completion := 0 ns;					-- by defn, nothing's running

		ELSE 	next_state := PowerDown;				-- Nothing happened

		END IF;
		TimeWH := 0 ns;							-- Clear the times which
		TimeWL := 0 ns;							-- check WE and CE pulse
		TimeEH := 0 ns;							-- widths
		TimeEL := 0 ns;

	WHEN OTHERS =>								-- If we got here, the main
		ASSERT (0>1)							-- state machine crashed.
		REPORT "Unrecognized main state achieved"			-- Won't happen.
		SEVERITY FAILURE;


END CASE;	-- END OF THE BIG STATE MACHINE

		-- **********************************************
		-- Everything below here gets executed every pass.
		-- **********************************************

	
state := next_state;							-- Update the big state machine
clock1 <= NOT(clock1) AFTER 0.5 ns;					-- Clock the other process



equal := '1';								-- Check if the address on the
FOR j IN 0 to MaxAddrLine LOOP						-- inputs equals the one in the
	IF hold_add(j) /= addr(j) THEN equal := '0';			-- latch
	END IF;
END LOOP;


IF timeEL >= timeWL THEN						-- If the last write was
									-- CE-controlled, then

	IF (equal = '0') AND (timeEL /= 0 ns) AND (ceb = '1') AND (ceb'LAST_EVENT < tEHAX)
 
	THEN 	IF AddrHoldFlag = '0' THEN				-- Check the Address Hold time
			ASSERT (0>1)					-- Gripe where needed
			REPORT "Violation of address hold time on write (tEHAX)"
			SEVERITY ERROR;
			AddrHoldFlag := '1';				-- Set a flag if AddrHold was
									-- violated
		END IF;
	ELSE	AddrHoldFlag := '0';					-- And clear the flag if it
	END IF;								-- wasn't violated.

	-- The flag serves to suppress all but the first instance of a particular error.
	-- Without that logic, every nanosecond there would be an assertion, till the
	-- hold period was complete.



ELSE									-- If the last write was 
									-- WE-controlled, then

	IF (equal = '0') AND (timeWL /= 0 ns) AND (web = '1') AND (web'LAST_EVENT < tWHAX) 
	THEN 	IF AddrHoldFlag = '0' THEN				-- Check the address hold time
			ASSERT (0>1)					-- Gripe where needed
			REPORT "Violation of address hold time on write (tWHAX)"
			SEVERITY ERROR;
			AddrHoldFlag := '1';				-- Set a flag if it was violated
		END IF;
	ELSE	AddrHoldFlag := '0';					-- and clear it if it wasn't.
	END IF;
END IF;

	
equal := '1';								-- Check if the data inputs have
FOR j IN 0 to 15 LOOP							-- changed since latching
	IF hold_data(j) /= dq(j) THEN equal := '0';
	END IF;
END LOOP;

IF timeEL >= timeWL THEN						-- If it's a CE-controlled write,

									-- The IF statement here breaks
									-- out to:
									-- 1) Data has changed since latch
									-- 2) Not immediately following
									--    powerup
									-- 3) CE is unasserted
									-- 4) CE has been unasserted less
									--    than the data hold time.

	IF (equal = '0') AND (timeEL /= 0 ns) AND (ceb = '1') AND (ceb'LAST_EVENT < tEHDX) 
	THEN 	IF DataHoldFlag = '0' THEN				-- Set a flag if Data Hold time
			ASSERT (0>1)					-- was violated, and gripe.
			REPORT "Violation of data hold time on write (tEHDX)"
			SEVERITY ERROR;
			DataHoldFlag := '1';
		END IF;
	ELSE	DataHoldFlag := '0';					-- Clear if it wasn't
	END IF;

ELSE									-- On WE-controlled write,
	IF (equal = '0') AND (timeWL /= 0 ns) AND (web = '1') AND (web'LAST_EVENT < tWHDX) 
	THEN 	IF DataHoldFlag = '0' THEN				-- Gripe and set flag when Data Hold
			ASSERT (0>1)					-- is violated
			REPORT "Violation of data hold time on write (tWHDX)"
			SEVERITY ERROR;
			DataHoldFlag := '1';
		END IF;
	ELSE	DataHoldFlag := '0';					-- and clear the flag when it's not
	END IF;
END IF;


IF NOT(((NOW - TimeSRValid) >= tQVVL) OR (VppSig = '1'))		-- If Vpp is out of range, and
THEN	IF VppholdFlag = '0' THEN					-- the device became free in the
		ASSERT (0>1)						-- last tQVVL, there's a Vpp Hold
		REPORT "Violation of Vpp Hold time on write (tQVVL)"	-- violation.
		SEVERITY ERROR;
		VppHoldFlag := '1';
	END IF;
ELSE	VppHoldFlag := '0';
END IF;


IF NOT ((Running = 0) OR (VccSig = '1'))				-- If Vcc is out of range, and the
THEN	IF VccMaintLFlag = '0' THEN					-- device is busy, that's bad.
		ASSERT (0>1)						
		REPORT "Vcc not maintained through latency period"
		SEVERITY ERROR;
		VccMaintLFlag := '1';
	END IF;
ELSE 	VccMaintLFlag := '0';   
END IF;


IF NOT ((SR(6) = '0' AND SR(2) = '0') OR (VccSig = '1'))		-- If Vcc is out of range, and
THEN	IF VccMaintSFlag = '0' THEN					-- something is suspended,
		ASSERT (0>1)						-- that's also bad.
		REPORT "Vcc not maintained through suspended operation"
		SEVERITY ERROR;
		VccMaintSFlag := '1';
	END IF;
ELSE	VccMaintSFlag := '0';
END IF;


IF NOT ((Running = 0) OR (VppSig = '1'))				-- If Vpp is out of range, and
THEN	IF VppMaintLFlag = '0' THEN					-- something is running, that's bad
		ASSERT (0>1)
		REPORT "Vpp not maintained through latency period"
		SEVERITY ERROR;
		VppMaintLFlag := '1';
	END IF;
ELSE 	VppMaintLFlag := '0';   
END IF;


IF NOT ((SR(6) = '0' AND SR(2) = '0') OR (VppSig = '1'))		-- If Vpp is out of range, and 
THEN	IF VppMaintSFlag = '0' THEN					-- something is suspended, that's 
		ASSERT (0>1)						-- bad, too.
		REPORT "Vpp not maintained through suspended operation"
		SEVERITY ERROR;
		VppMaintSFlag := '1';
	END IF;
ELSE	VppMaintSFlag := '0';
END IF;


-- This code runs when something gets finished (as determined by the time we wrote in the Completion
-- variable when the something in question started.

IF ((Completion /= 0 ns) AND (NOW >= Completion)) THEN			-- a latency just finished

	TimeSRValid := NOW;						-- Note when the SR became valid.
									-- A couple hold times need this,
									-- theoretically.
	SR(7) := '1';							-- Set the Dev Free flag
	IF (STSMode = "00") THEN sts <= 'Z';				-- If STS is in RY/BY, mark it ready
	END IF;

	IF ((STSMode(1) = '1') AND ((Running = WriteProgram)		-- If the STS mode involves pulsing
									-- at program completion, and a 
									-- program is what just completed,

					  OR (Running = WriteBufNum) OR (Running = WriteBufData)
					  OR (Running = WriteBufCfrm))) THEN
		sts <= TRANSPORT '0';
		sts <= TRANSPORT 'Z' after 250 ns;			-- then pulse.
	END IF;


	IF ((STSMode(0) = '1') AND ((Running = WriteBlkErase)		-- If the STS mode involves pulsing
									-- at erase completion, and an
									-- erase just completed,
					  OR (Running = WriteFullChipErase))) THEN

		sts <= TRANSPORT '0';					-- then pulse.
		sts <= TRANSPORT 'Z' after 250 ns;
	END IF;
	
	IF (running = WriteBufCfrm) THEN
		ActiveBuffer := 0;
		FOR j IN 1 TO NumBuffers LOOP
			IF BufStatus(j) = BufWriting THEN ActiveBuffer := j;
			END IF;
		END LOOP;
		BufStatus(ActiveBuffer) := BufFree;
	END IF;			

	Running := 0;							-- Nothing is running,
	Completion := 0 ns;						-- so there's no time at which it 
									-- will complete.
	ActiveBuffer := 0;
	FOR j IN 1 TO NumBuffers LOOP
		IF BufStatus(j) = BufFull THEN ActiveBuffer := j;
		END IF;
	END LOOP;

	IF (ActiveBuffer /= 0) THEN	
		-- There's a buffer full and queued to write
		-- Place code to start it writing here.
		-- This code should be plagiarized from the WriteBufCfrm subcase.


		-- This is just terrible form - the following code over to the right is a carbon copy
		-- of the code in the WriteBufCfrm case.  In a perfect model, it would be off in a 
		-- function somewhere, but that's not how I chose to do it here.


						IF (BufbytebLatch(ActiveBuffer) = '1') THEN
										
										-- If it was all done in 
										-- word mode...

							FOR j IN 0 TO BufCount(ActiveBuffer) LOOP

										-- Check if each program can
										-- be done without having
										-- to erase the memory.
										-- We cheat, though - the 
										-- loop always says you can't
										-- program if there's already
										-- a program fail in the SR.

								equal := '0';

								IF SR(4) = '0' THEN equal := '1';	
								END IF;

								FOR k IN 0 TO 15 LOOP
									IF ((WriteBuffers(ActiveBuffer,j)(k) AND 
									(NOT(memory(BufStartAddr
									(ActiveBuffer) + j)(k)))) 
								 	= '1') THEN equal := '0';
									END IF;
								END LOOP;

										-- If we can program, and
										-- nothing failed already, 
										-- do program.

								IF (equal = '1')
								THEN 	memory(BufStartAddr(ActiveBuffer) + j) :=
									WriteBuffers(ActiveBuffer, j);

										-- If something failed,
										-- set the prog fail flag.

								ELSE 	SR(4) := '1';
								END IF;
							END LOOP;

										-- Note the buffer is writing	
										-- to the flash,

							BufStatus(ActiveBuffer) := BufWriting;

										-- that a buffer write is
										-- why the device is busy,
							running := WriteBufCfrm;

										-- that the next write will
										-- be a command,
							WriteType := WriteCmd;

										-- that the next read will 
										-- be of the SR,
							ReadType := ReadStatReg;

										-- when we expect to finish,
							Completion := NOW + ((BufCount(ActiveBuffer) + 1) 
										* W16_ProgWordWBuffer);
						
										-- that the device is not
										-- ready,
							SR(7) := '0';

										-- and if there are any free
										-- write buffers left. 
							XSR(7) := '0';
							FOR j IN 1 TO NumBuffers LOOP
								IF BufStatus(j) = BufFree THEN XSR(7) := '1';
								END IF;
							END LOOP;


										-- Otherwise, the write was
										-- started in byte mode, and
										-- things get much more
										-- complicated.

						-- The byte mode version of the write to flash break into
						-- three parts.

						-- 1) If the starting address is odd, the write starts by 
						--    writing to only the high byte of the first address.
						-- 2) The middle portion of the write goes a word at a time.
						-- 3) If the ending address is even, the write ends by
						--    writing to only the low byte of the ending address.


										-- This is the first part.
						ELSE	IF (BufStartAddr(ActiveBuffer) mod 2 = 1) THEN

										-- This loop checks both
										-- if any program errors
										-- have occurred and if the
										-- current byte can be 
										-- programmed without erasing

								equal := '1';
								IF SR(4) = '1' THEN equal := '0';
								END IF;

								FOR k IN 8 TO 15 LOOP
									IF (WriteBuffers(ActiveBuffer, 0)(k) AND
										NOT(memory(BufStartAddr
										(activeBuffer))(k)))
										 = '1' THEN equal := '0';
									END IF;
								END LOOP;

										-- And this loop programs the
										-- high byte

								IF equal = '1' THEN
									FOR k IN 8 TO 15 LOOP
										memory(BufStartAddr
										(ActiveBuffer))(k) := 
										WriteBuffers(ActiveBuffer, 0)(k);
									END LOOP;
								ELSE	SR(4) := '1';
								END IF;
								j_start := 1;	-- This notes where in the
										-- buffer to begin the middle
										-- portion of the write

							ELSE	j_start := 0;	-- So does this.
							END IF;

										-- This logic notes at what
										-- word of the buffer to end
										-- the middle portion of the
										-- write.
							
							IF BufEndAddr(ActiveBuffer) mod 2 = 0 THEN
								IF j_start = 1 THEN
									j_end := BufCount(ActiveBuffer) / 2 - 1;
								ELSE 	j_end := ((BufCount(ActiveBuffer) 
											- 1) / 2) - 1;
								END IF;
							ELSE	IF j_start = 1 THEN
									j_end := (BufCount(ActiveBuffer) - 1) 
											/ 2;
								ELSE	j_end := BufCount(ActiveBuffer) / 2 - 1;
								END IF;	
						END IF;


										-- This performs the middle
										-- portion of the write.
							FOR j IN j_start TO j_end LOOP


										-- This loop checks both
										-- if any program errors
										-- have occurred and if the
										-- current byte can be 
										-- programmed without erasing
								equal := '1';
								IF SR(4) = '1' THEN equal := '0';
								END IF;

								FOR k IN 0 TO 15 LOOP
									IF (WriteBuffers(ActiveBuffer, j)(k) AND
									NOT(memory(BufStartAddr(ActiveBuffer) + j)
									(k))) = '1' THEN equal := '0';
									END IF;
								END LOOP;

										-- This loop programs the
										-- middle words.
									
								IF equal = '1' THEN
									FOR k IN 0 TO 15 LOOP
										memory(BufStartAddr(ActiveBuffer)
											 + j)(k) := 
										WriteBuffers(ActiveBuffer, j)(k);
									END LOOP;
								ELSE	SR(4) := '1';
								END IF;
							END LOOP;


										
										-- This is the last part of
										-- the write.
							IF (BufEndAddr(ActiveBuffer) mod 2 = 0) THEN


										-- This loop checks both
										-- if any program errors
										-- have occurred and if the
										-- current byte can be 
										-- programmed without erasing
								j_end := j_end + 1;
								equal := '1';
								IF SR(4) = '1' THEN equal := '0';
								END IF;

								FOR k IN 0 TO 7 LOOP
									IF (WriteBuffers(ActiveBuffer, j_end)
									(k) AND NOT(memory(BufStartAddr
									(ActiveBuffer) + j_end)(k))) = '1' 
									THEN 	equal := '0';
									END IF;
								END LOOP;

										-- This programs the 
										-- straggler low byte

								IF equal = '1' THEN
									FOR k IN 0 TO 7 LOOP
										memory(BufStartAddr(ActiveBuffer)
											+ j_end)(k) := 
										WriteBuffers(ActiveBuffer, j_end)
											(k);
									END LOOP;
								ELSE	SR(4) := '1';
								END IF;
						
							END IF;	
						


										-- Note the buffer is writing	
										-- to the flash,
							BufStatus(ActiveBuffer) := BufWriting;

										-- that a buffer write is
										-- why the device is busy,
							running := WriteBufCfrm;

										-- that the next write will
										-- be a command,
							WriteType := WriteCmd;

										-- that the next read will 
										-- be of the SR,
							ReadType := ReadStatReg;

										-- when we expect to finish,
							Completion := NOW + ((BufCount(ActiveBuffer) + 1) 
										* W16_ProgByteWBuffer);
						
										-- that the device is not
										-- ready,
							SR(7) := '0';

										-- and if there are any free
										-- write buffers left. 
							XSR(7) := '0';
							FOR j IN 1 TO NumBuffers LOOP
								IF BufStatus(j) = BufFree THEN XSR(7) := '1';
								END IF;
							END LOOP;
						END IF;




	END IF;


END IF;

END PROCESS MAIN;




EL_GUAPO:PROCESS

	-- This process exists solely to clock the main process.

BEGIN

	WAIT ON clock1;
		clock2 <= not clock2 after 0.5 ns;
END PROCESS EL_GUAPO;


END behavior;
